"""
Created on 2019年6月28日
@author: 刘益宗
第三方cache库
"""
import copy
import time
from collections import OrderedDict
from copy import deepcopy
from decimal import Decimal
from threading import RLock

_NOTSET = object()


class Cache(object):
    """
    缓存类
    """
    def __init__(self, maxsize=None, ttl=None, timer=None, default=None):
        """
        初始化函数
        :param maxsize: 缓存最大数量
        :param ttl: 默认过期时间
        :param timer:  计时器
        :param default: 缓存默认值
        """
        if maxsize is None:
            maxsize = 256

        if ttl is None:
            ttl = 0

        if timer is None:
            timer = time.time

        self.setup()
        self.configure(maxsize=maxsize, ttl=ttl, timer=timer, default=default)

    def setup(self):
        """
        缓存初始化
        :return:
        """
        self._cache = OrderedDict()  # 缓存对象
        self._expire_times = {}
        self._lock = RLock()

    def configure(self, maxsize=None, ttl=None, timer=None, default=None):
        """
        缓存配置
        """
        if maxsize is not None:
            if not isinstance(maxsize, int):
                raise TypeError("maxsize must be an integer")

            if not maxsize >= 0:
                raise ValueError("maxsize must be greater than or equal to 0")

            self.maxsize = maxsize

        if ttl is not None:
            if not isinstance(ttl, (int, float, Decimal)):
                raise TypeError("ttl must be a number")

            if not ttl >= 0:
                raise ValueError("ttl must be greater than or equal to 0")

            self.ttl = ttl

        if timer is not None:
            if not callable(timer):
                raise TypeError("timer must be a callable")

            self.timer = timer

        self.default = default

    def __repr__(self):
        return "{}({})".format(self.__class__.__name__, list(self.copy().items()))

    def __len__(self):
        with self._lock:
            return len(self._cache)

    def __contains__(self, key):
        with self._lock:
            return key in self._cache

    def __iter__(self):
        yield from self.keys()

    def __next__(self):
        return next(iter(self._cache))

    def copy(self):
        """
        Return a copy of the cache.

        Returns:
            OrderedDict
        """
        with self._lock:
            return self._cache.copy()

    def keys(self):
        """
        Return ``dict_keys`` view of all cache keys.

        Note:
            Cache is copied from the underlying cache storage before returning.

        Returns:
            dict_keys
        """
        return self.copy().keys()

    def values(self):
        """
        Return ``dict_values`` view of all cache values.

        Note:
            Cache is copied from the underlying cache storage before returning.

        Returns:
            dict_values
        """
        return self.copy().values()

    def items(self):
        """
        Return a ``dict_items`` view of cache items.

        Warning:
            Returned data is copied from the cache object, but any modifications to
            mutable values will modify this cache object's data.

        Returns:
            dict_items
        """
        return self.copy().items()

    def clear(self):
        """Clear all cache entries."""
        with self._lock:
            self._clear()
        assert len(self) == 0

    def _clear(self):
        self._cache.clear()
        self._expire_times.clear()

    def size(self):
        """Return number of cache entries."""
        return len(self)

    def full(self):
        """
        Return whether the cache is full or not.

        Returns:
            bool
        """
        if self.maxsize == 0:
            return False
        return len(self) >= self.maxsize

    def has(self, key):
        """
        Return whether cache key exists and hasn't expired.

        Returns:
            bool
        """
        with self._lock:
            return self._has(key)

    def _has(self, key):
        # Use get method since it will take care of evicting expired keys.
        return self._get(key, default=_NOTSET) is not _NOTSET

    def try_get_value(self, key, ttl, func, *args, **kwargs):
        # return func(*args, **kwargs)
        #log_info("Key:%s", key)
        rs = self.get(key)
        if not rs:
            #log_info("缓存数据失效,从函数%s中重新获取数据,重置缓存",func)
            rs = func(*args, **kwargs)
            if rs is None:
                rs = {'status': 999, 'msg': 'UnknownErr', 'data': {}}
            elif rs['status'] == 0 and rs['data']:
                self.set(key, rs, ttl)
                rs = deepcopy(rs)
        return rs



    def get(self, key, default=None):
        """
        Return the cache value for `key` or `default` or ``missing(key)`` if it doesn't
        exist or has expired.

        Args:
            key (mixed): Cache key.
            default (mixed, optional): Value to return if `key` doesn't exist. If any
                value other than ``None``, then it will take precendence over
                :attr:`missing` and be used as the return value. If `default` is
                callable, it will function like :attr:`missing` and its return value
                will be set for the cache `key`. Defaults to ``None``.

        Returns:
            mixed: The cached value.
        """
        with self._lock:
            return self._get(key, default=default)



    def _get(self, key, default=None):
        try:
            value = self._cache[key]
            if self.expired(key):
                self._delete(key)
                raise KeyError
            value = deepcopy(value)
        except KeyError:
            if default is None:
                default = self.default

            # if callable(default):
            #     value = default(key)
            #     value = deepcopy(value)
            #     self._set(key, value)
            # else:
                value = default

        return value

    # def add(self, key, value, ttl=None):
    #     """
    #     Add cache key/value if it doesn't already exist. Essentially, this method
    #     ignores keys that exist which leaves the original TTL in tact.
    #
    #     Note:
    #         Cache key must be hashable.
    #
    #     Args:
    #         key (mixed): Cache key to add.
    #         value (mixed): Cache value.
    #         ttl (int, optional): TTL value. Defaults to ``None`` which uses :attr:`ttl`.
    #     """
    #     with self._lock:
    #         self._add(key, value, ttl=ttl)

    def _add(self, key, value, ttl=None):
        # if not isinstance(key, str):
        #     raise Exception('Key Must be Str')

        if self._has(key):
            return
        self._set(key, value, ttl=ttl)

    def set(self, key, value, ttl=None):
        """
        Set cache key/value and replace any previously set cache key. If the cache key
        previous existed, setting it will move it to the end of the cache stack which
        means it would be evicted last.

        Note:
            Cache key must be hashable.

        Args:
            key (mixed): Cache key to set.
            value (mixed): Cache value.
            ttl (int, optional): TTL value. Defaults to ``None`` which uses :attr:`ttl`.
        """
        with self._lock:
            self._set(key, value, ttl=ttl)
            assert self.get(key) is not None

    def _set(self, key, value, ttl=None):
        # if not isinstance(key, str):
        #     raise Exception('Key Must be Str')

        if ttl is None:
            ttl = self.ttl

        if key not in self:
            self.evict()

        self._delete(key)
        self._cache[key] = value

        if ttl and ttl > 0:
            self._expire_times[key] = self.timer() + ttl

    def delete(self, key):
        """
        Delete cache key and return number of entries deleted (``1`` or ``0``).

        Returns:
            int: ``1`` if key was deleted, ``0`` if key didn't exist.
        """
        with self._lock:
            rs = self._delete(key)
            assert self.get(key) is None
            return rs

    def _delete(self, key):
        count = 0

        try:
            del self._cache[key]
            count = 1
        except KeyError:
            pass

        try:
            del self._expire_times[key]
        except KeyError:
            pass

        return count

    def delete_expired(self):
        """
        Delete expired cache keys and return number of entries deleted.

        Returns:
            int: Number of entries deleted.
        """
        with self._lock:
            return self._delete_expired()

    def _delete_expired(self):
        count = 0

        if not self._expire_times:
            return count

        # Use a static expiration time for each key for better consistency as opposed to
        # a newly computed timestamp on each iteration.
        expires_on = self.timer()
        expire_times = self._expire_times.copy()

        for key, expiration in expire_times.items():
            if expiration <= expires_on:
                count += self._delete(key)

        return count

    def expired(self, key, expires_on=None):
        """
        Return whether cache key is expired or not.

        Args:
            key (mixed): Cache key.
            expires_on (float, optional): Timestamp of when the key is considered
                expired. Defaults to ``None`` which uses the current value returned from
                :meth:`timer`.

        Returns:
            bool
        """
        if not expires_on:
            expires_on = self.timer()

        try:
            return self._expire_times[key] <= expires_on
        except KeyError:
            return key not in self

    def expire_times(self):
        """
        Return cache expirations for TTL keys.

        Returns:
            dict
        """
        with self._lock:
            return self._expire_times.copy()

    def evict(self):
        """
        Perform cache eviction per the cache replacement policy:

        - First, remove **all** expired entries.
        - Then, remove non-TTL entries using the cache replacement policy.

        When removing non-TTL entries, this method will only remove the minimum number
        of entries to reduce the number of entries below :attr:`maxsize`. If
        :attr:`maxsize` is ``0``, then only expired entries will be removed.

        Returns:
            int: Number of cache entries evicted.
        """
        count = self.delete_expired()

        if not self.full():
            return count

        with self._lock:
            while self.full():
                try:
                    self._popitem()
                except KeyError:  # pragma: no cover
                    break
                count += 1

        return count

    def popitem(self):
        """
        Delete and return next cache item, ``(key, value)``, based on cache replacement
        policy while ignoring expiration times (i.e. the selection of the item to pop is
        based soley on the cache key ordering).

        Returns:
            tuple: Two-element tuple of deleted cache ``(key, value)``.
        """
        with self._lock:
            self._delete_expired()
            return self._popitem()

    def _popitem(self):
        try:
            key = next(self)
        except StopIteration:
            raise KeyError("popitem(): cache is empty")

        value = self._cache[key]

        self._delete(key)

        return (key, value)

    def try_get_value_v2(self, key, ttl, func, *args, **kwargs):
        rs = self.get_v2(key)
        if not rs:
            # log_info("缓存数据失效,从函数%s中重新获取数据,重置缓存",func)
            rs = func(*args, **kwargs)
            if rs is None:
                rs = {'status': 999, 'msg': 'UnknownErr', 'data': {}}
            elif rs['status'] == 0 and rs['data']:
                self.set(key, rs, ttl)
                rs = deepcopy(rs)
        return rs

    def get_v2(self, key, default=None):
        with self._lock:
            return self._get_v2(key, default=default)

    def _get_v2(self, key, default=None):
        try:
            value = self._cache[key]
            if self.expired(key):
                self._delete(key)
                raise KeyError
            #浅拷贝
            value = copy.copy(value)
        except KeyError:
            if default is None:
                default = self.default

                value = default

        return value

    # def delete_prefix(self, prefix):
    #     count = 0
    #     with self._lock:
    #         for key in self.keys():
    #             if key.startswith(prefix):
    #                 count += self.delete(key)
    #                 assert self.get(key) is None
    #     return count

    # def delete_many(self, keys):
    #     count = 0
    #     with self._lock:
    #         for key in keys:
    #             count += self.delete(key)
    #             assert self.get(key) is None
    #     return count

class CacheManager:

    def __init__(self):
        self.DeviceCache = Cache()
        self.InputCache = Cache()
        self.OutputCache = Cache()
        self.BkgCache = Cache()
        self.LayerCache = Cache()
        self.MvrCache = Cache()
        self.MvrWindowCache = Cache()
        self.PresetCache = Cache()
        self.ScreenCache = Cache()
        self.UserCache = Cache()
        self.IPCCache = Cache()
        self.SecurityCache = Cache()
        self.VerifyCodeCache = Cache()
        self.TokenCache = Cache()
        self.RoleCache = Cache()
        self.NdiCache = Cache()

    def clear_all(self):
        self.DeviceCache.clear()
        self.InputCache.clear()
        self.OutputCache.clear()
        self.BkgCache.clear()
        self.LayerCache.clear()
        self.MvrCache.clear()
        self.MvrWindowCache.clear()
        self.PresetCache.clear()
        self.ScreenCache.clear()
        self.UserCache.clear()
        self.RoleCache.clear()
        self.IPCCache.clear()
        self.SecurityCache.clear()
        self.NdiCache.clear()


    def set_login_token(self, key, value):
        tokens = []
        for k, v in cacher.TokenCache.items():
            if v['id'] == value['id']:
                cacher.TokenCache.delete(k)
                tokens.append(k)
        cacher.TokenCache.set(key, value)
        return tokens



cacher = CacheManager()


if __name__ == '__main__':
    for x in range(10):
        cacher.DeviceCache.set(x, x)
        cacher.InputCache.set(x, x)

    cacher.DeviceCache.get(1)
    cacher.DeviceCache.delete(1)
    cacher.clear_all()
    print(len(cacher))
