import re
import threading

from flask import g
from itsdangerous import TimedJSONWebSignatureSerializer as Serializer

from H_9U.dao.device_info_dao import deviceInfodao
from H_9U.dao.role import roledao
from H_9U.dao.auth_code import auth_code_dao
from werkzeug.security import check_password_hash, generate_password_hash
from H_9U.util.assembly_data import remove_element_in_place
from H_9U.util.common import device_safe_config_read
from H_9U.api.filter.filter import role_filter_data_func
from H_9U.util.common import md5, compare_elements
from H_9U.dao.user import userdao
from H_9U.conf.syssettings import SysSettings
from H_9U.models.result import get_result_model, ResInfo
from H_9U.util.log import logger, log_info
from H_9U.util.cache import cacher
from H_9U.models.cachekey import CacheKey, CacheDuration
from H_9U.models.sysconst import Auth_Code_Expire_Time, UserStatus, SafeConfig, SecuritySessionExpiredType
import pickle
import base64
import time
from H_9U.util.regular_check import regular_check_phone, regular_check_email, regular_check_username, \
    regular_check_password
from H_9U.util.verification_code import draw_image


class UserApi:

    def _get_by_id(self, user_id, with_pwd=False):
        """
        获取用户基本信息并加入缓存
        :param user_id: 用户id
        :return: 用户基本信息
        """
        return userdao.get_by_id(user_id)

    def __user_func_auth(self, funcs, u_fns, is_admin):
        """
        用户权限匹配
        :param funcs: 全部权限
        :param u_fns: 用户权限
        :param is_admin: 是否超级管理员
        :return:
        """
        if funcs:
            for fn in funcs:
                if is_admin:
                    fn['authed'] = 1
                else:
                    if fn['id'] in u_fns:
                        fn['authed'] = 1
                    else:
                        fn['authed'] = 0
                if fn.get('funcs'):
                    self.__user_func_auth(fn.get('funcs'), u_fns, is_admin)

    def _get_by_username(self, username):
        """
        根据用户名获取用户
        :param username: 用户名
        :return: 用户信息
        """
        return userdao.new_get_by_username(username)

    def _insert(self, username, password, tel, email, status, role_id):
        """
        插入用户
        清除用户列表缓存
        :param username: 用户名
        :param password: 密码
        :param tel: 电话
        :param email: 邮箱
        :param active: 状态
        :param permissions: 权限
        :return: 用户id
        """
        return userdao.insert_user(
            username, password, tel, email, status, role_id)

    def _update(self, user_id, role_id, user_params):
        """
        修改用户
        清除用户列表缓存
        清除用户信息缓存
        清除用户权限缓存
        请求用户信息和权限缓存
        :param username: 用户名
        :param password: 密码
        :param tel: 电话
        :param email: 邮箱
        :param active: 状态
        :param permissions: 权限
        :return: 结果对象
        """
        return userdao.update(user_id, role_id, user_params)

    def _get_all_user(self):
        """
        获取全部用户并加入缓存
        :return: 结果
        """
        return userdao.get_all_user()

    def _delete(self, user_id):
        """
        删除用户
        删除用户列表缓存
        删除用户详情缓存
        删除用户权限缓存
        删除用户信息和权限缓存
        :param user_id: id
        :return: 结果对象
        """
        return userdao.get_by_id(user_id)

    def _get_all_func(self):
        """
        获取系统全部权限并加入缓存
        :return: 结果
        """
        def __get_all_func():
            rs = userdao.get_all_func()
            if rs['status'] == 0:
                funcs = [x for x in rs['data'] if x['level'] == 0]
                for f in funcs:
                    f['funcs'] = [
                        x for x in rs['data'] if x['parentId'] == f['id']]
                rs['data'] = funcs
            return rs

        key = CacheKey.func_list()
        return cacher.UserCache.try_get_value(
            key, CacheDuration.Long, __get_all_func)

    def _update_password(self, user_id, password, active):
        rs = userdao.update_password(user_id, password, active)
        if rs['status'] == 0:
            cacher.UserCache.delete(CacheKey.user_detail(user_id))
            cacher.UserCache.delete(CacheKey.user_list())
        return rs

class UserSvc(UserApi):
    def __init__(self):
        self.token_data = dict()
        self.login_data = dict()
        self.__delay_time = 60
        self.login_config = dict()
        self.pwd_config = dict()
        self.permission = dict()
        self.is_reset = dict()
        self.init_config()
        # 定时调度:用于Socket重链
        self.timer = threading.Timer(self.__delay_time, self.__clear_expire)
        self.timer.start()

    def init_config(self):
        user_config_rs = self.user_security_read(0)
        if user_config_rs['status'] == 0:
            self.login_config = user_config_rs['data']['loginConfig']
            if self.login_config['sessionExpireType'] == SecuritySessionExpiredType.UnOverdue:
                return
            if self.pwd_config and self.compare_pwd_config(self.pwd_config, user_config_rs['data']['pwdConfig']):
                self.token_data.clear()
            self.pwd_config = user_config_rs['data']['pwdConfig']
            if self.login_config['loginFailedCheck']['enable'] == 0:
                self.login_data.clear()

    def compare_pwd_config(self, old, new):
        flag = False
        if (new['pwdLength']['min'] > old['pwdLength']['min'] or new['pwdLength']['max'] < old['pwdLength']['max']) or \
                new['pwdRule']['nums'] > old['pwdRule']['nums'] or \
                new['pwdRule']['lowerLetter'] > old['pwdRule']['lowerLetter'] or \
                new['pwdRule']['upperLetter'] > old['pwdRule']['upperLetter'] or \
                new['pwdRule']['symbolCharacter'] > old['pwdRule']['symbolCharacter']:
            flag = True
        return flag

    def __clear_expire(self):
        """清空所有过期数据，包含登录和token"""
        if self.login_config['sessionExpireType'] == SecuritySessionExpiredType.UnOverdue:
            return
        expire_time = self.login_config['sessionExpireTime'] * 60
        try:
            time_key = 'login_time' if self.login_config['sessionExpireType'] == 1 else 'last_time'
            expires = []
            for k in self.token_data.keys():
                if time.monotonic() - self.token_data[k][time_key] > expire_time:
                    expires.append(k)
            for k in expires:
                self.token_data.pop(k)
        except Exception as e:
            logger.error(e)
        try:
            if self.login_config['loginFailedCheck']['enable'] == 1:
                expire_time = self.login_config['loginFailedCheck']['duration'] * 60
                expires = []
                for k in self.login_data.keys():
                    if time.monotonic() - \
                            self.login_data[k]['first_time'] > expire_time:
                        expires.append(k)
                for k in expires:
                    self.login_data.pop(k)
        except Exception as e:
            logger.error(e)
        self.timer = threading.Timer(self.__delay_time, self.__clear_expire)
        self.timer.start()

    def __get_login_data(self, username):
        """获取登录用户的登录次数信息，没有初始化默认值"""
        login_info = self.login_data.get(username, None)
        first_login_info = {
            'count': 0,
            'last_time': time.monotonic(),
            'first_time': time.monotonic()}
        # 未登录记录登录次数
        if not login_info:
            login_info = first_login_info
        # 判断登录时间间隔
        if self.login_config['loginFailedCheck']['enable'] == 1:
            if time.monotonic() - login_info['first_time'] > self.login_config['loginFailedCheck']['duration'] * 60:
                self.login_data.pop(username)  # 超时则移除
                login_info = first_login_info
        else:
            login_info = first_login_info
        return login_info

    def __valid_login_times(self, login_info, params):
        """检验登录次数和验证码"""
        # 启用失败校验
        if self.login_config['loginFailedCheck']['enable'] == 1:
            # 判断锁定次数
            if login_info['count'] > self.login_config['loginFailedCheck']['lockUserTimes']:
                rs = get_result_model(ResInfo.User_Login_Locked)
                lock_time = self.login_config['loginFailedCheck']['duration'] * \
                            60 - (time.monotonic() - login_info['first_time'])
                rs['data'] = {'lockTime': lock_time}
                return rs
            # 判断验证码
            if login_info['count'] > self.login_config['loginFailedCheck']['showVerifyCodeTimes']:
                verify_id = params.get('verifyId')
                verify_code = params.get('verifyCode')
                if verify_id and verify_code:
                    if not self.__check_verify_code(verify_id, verify_code):
                        return get_result_model(ResInfo.User_Verify_Code_Err)
                else:
                    return get_result_model(
                        ResInfo.User_Login_Err_Need_Verify_Code)
        return get_result_model()

    def valid_user_password(self, pwd):
        """校验用户密码是否合规"""
        if self.pwd_config['pwdLength']['min'] > len(pwd):
            return False
        if self.pwd_config['pwdLength']['max'] < len(pwd):
            return False
        if self.pwd_config['pwdRule']['nums'] and not re.match('.*\\d+.*', pwd):
            return False
        if self.pwd_config['pwdRule']['lowerLetter'] and not re.match('.*[a-z]+.*', pwd):
            return False
        if self.pwd_config['pwdRule']['upperLetter'] and not re.match('.*[A-Z]+.*', pwd):
            return False
        if self.pwd_config['pwdRule']['symbolCharacter'] and not re.match(r'.*[^a-zA-Z0-9]+.*', pwd):
            return False
        return True

    def __check_verify_code(self, verify_id, code):
        """
        校验验证码
        :param verify_id: 验证码id
        :param code: 验证码
        :return: 结果
        """
        text = cacher.VerifyCodeCache.get(verify_id)
        if text is not None and code.lower() == text.lower():
            return True
        else:
            return False

    def login(self, username, password, params):
        """
        用户登录
        :param username: 用户名
        :param password: 密码
        :return: token，deviceId
        """
        is_safe = self.__get_safe_config()
        g.safe = is_safe
        if is_safe == SafeConfig.IsNotSafe:
            return self.common_login(username, password, is_safe)
        else:
            login_info = self.__get_login_data(username)
            login_info['count'] += 1
            login_info['last_time'] = time.monotonic()
            rs = self.__valid_login_times(login_info, params)
            if self.login_config['loginFailedCheck']['enable'] == 1:
                self.login_data[username] = login_info
            if rs['status'] != 0:
                return rs
            rs = self._get_by_username(username)
            if not rs['data']:
                return get_result_model(ResInfo.User_Not_Exist)
            user = rs['data']
            login_info_rs = self.__build_login_info(is_safe, user, password, login_info)
            if login_info_rs['status'] != 0:
                return login_info_rs
            token = self.__build_token_data(is_safe, user)
            if user['status'] == UserStatus.Locked:  # 用户锁定
                return get_result_model(ResInfo.User_Not_Active)
            user_id = user['id']
            if not self.valid_user_password(password):
                rs = userdao.update_user_status(user_id, UserStatus.PasswordChanged)
                if rs['status'] == 0:
                    user['status'] = UserStatus.PasswordChanged
            # 多端登录
            if self.login_config['multiLoginAllow'] != 1:
                self.delete_user_token(user_id)
            user_permission = self.get_permission_list(
                user_id)
            self.permission[user_id] = user_permission
            # key = CacheKey.user_permissions(user_id)
            # cacher.UserCache.set(key, user_permission)
            # 加入到登录用户中
            login_info('用户 %s 登录，生成token： %s' % (user['username'], token))
            self.token_data[token] = {
                'userId': user_id,
                'ip': g.ip,
                'username': user['username'],
                'login_time': time.monotonic(),
                'last_time': time.monotonic(),
                'system_login_time': time.time(),
                'system_last_time': time.time()}
            # 登录成功，删除登录数据
            if username in self.login_data:
                self.login_data.pop(username)
            funcs = []
            role_id = 2
            rs = roledao.get_role_by_user_id(user_id)
            if rs['status'] == 0 and rs['data']:
                role_id = rs['data']['roleId']
            user_rs = self.get_with_func(role_id)
            if user_rs['status'] == 0:
                funcs = user_rs['data']
            rs = get_result_model()
            rs['data'] = {
                'token': token,
                'userId': user['id'],
                'type': user['type'],
                'email': user['email'],
                'status': user['status'],
                'roleId': role_id,
                'userName': user['username'],
                'isAdmin': user['superadmin'],
                'deviceId': SysSettings.Device_Id if SysSettings.Device_Id else 0,
                "userFuncs": funcs}
            return rs

    def common_login(self, username, password, is_safe):
        rs = self._get_by_username(username)
        if not rs['data']:
            return get_result_model(ResInfo.User_Not_Exist)
        user = rs['data']
        if user['password'] != md5(password):
            return get_result_model(ResInfo.User_Password_Error)
        token = self.__build_token_data(is_safe, user)
        if user['status'] == UserStatus.Locked:  # 用户锁定
            return get_result_model(ResInfo.User_Not_Active)
        user_id = user['id']
        user_permission = self.get_permission_list(
            user_id)
        self.permission[user_id] = user_permission
        # key = CacheKey.user_permissions(user_id)
        # cacher.UserCache.set(key, user_permission)
        # 加入到登录用户中
        log_info('用户 %s 登录，生成token： %s' % (user['username'], token))
        self.token_data[token] = {
            'userId': user_id,
            'ip': g.ip,
            'username': user['username'],
            'login_time': time.monotonic(),
            'last_time': time.monotonic(),
            'system_login_time': time.time(),
            'system_last_time': time.time()}
        funcs = []
        role_id = 2
        rs = roledao.get_role_by_user_id(user_id)
        if rs['status'] == 0 and rs['data']:
            role_id = rs['data']['roleId']
        user_rs = self.get_with_func(role_id)
        if user_rs['status'] == 0:
            funcs = user_rs['data']
        rs = get_result_model()
        rs['data'] = {
            'token': token,
            'userId': user['id'],
            'type': user['type'],
            'email': user['email'],
            'status': user['status'],
            'roleId': role_id,
            'userName': user['username'],
            'isAdmin': user['superadmin'],
            'deviceId': SysSettings.Device_Id if SysSettings.Device_Id else 0,
            "userFuncs": funcs}
        return rs

    def __build_token_data(self, is_safe, user):
        token = None
        if is_safe == SafeConfig.IsSafe:
            token = self._generate_safe_token(user['id'])
        elif is_safe == SafeConfig.IsNotSafe:
            token = self._generate_auth_token(user_id=user['id'], pwd=user['password'])
        return token

    def __build_login_info(self, is_safe, user, password, login_info):
        rs = get_result_model()
        if is_safe == SafeConfig.IsSafe:
            flag = check_password_hash(user['password'], password)
            if not user or not flag:
                if self.login_config['loginFailedCheck']['enable'] == 1 and login_info['count'] == \
                        self.login_config['loginFailedCheck']['showVerifyCodeTimes']:
                    return get_result_model(ResInfo.User_Login_Err_Need_Verify_Code)
                return get_result_model(ResInfo.User_Login_Err)
        elif is_safe == SafeConfig.IsNotSafe:
            if not user or user['password'] != md5(password):
                if self.login_config['loginFailedCheck']['enable'] == 1 and login_info['count'] == \
                        self.login_config['loginFailedCheck']['showVerifyCodeTimes']:
                    return get_result_model(ResInfo.User_Login_Err_Need_Verify_Code)
                return get_result_model(ResInfo.User_Login_Err)
        return rs

    def __get_safe_config(self):
        from H_9U.util.cache import cacher
        device_id = 0
        key = CacheKey.device_is_safe(device_id)
        is_safe = cacher.DeviceCache.get(key)
        if is_safe is None:
            is_safe = device_safe_config_read(SysSettings.Middleware_Safe_Config_Path)
        return is_safe

    def delete_user_token(self, user_id):
        """删除以登录用户的token信息，退出登录"""
        login_users = []
        for k in self.token_data.keys():
            if self.token_data[k]['userId'] == user_id:
                login_users.append(k)
        for k in login_users:
            self.token_data.pop(k)

    def get_with_func(self, role_id):
        """
        获取用户基本信息和权限信息
        :param role_id: role_id
        :return: 用户和权限信息
        """
        key = CacheKey.role_func(role_id)
        rs = cacher.RoleCache.get(key, None)
        if not rs:
            from H_9U.service.role import rolesvc
            role_rs = rolesvc.get_role_by_id(role_id)
            if role_rs['status'] == 0 and role_rs['data']:
                role_type = role_rs['data']['role_type']
                if role_type == 1:
                    des = roledao.get_funcs()
                    return des
                else:
                    fn_rs = roledao.get_func_by_role_id(role_id)
                    rs_menu = []
                    rs = get_result_model()
                    if fn_rs['status'] == 0:
                        if fn_rs['data']:
                            for fn_rs_message in fn_rs['data']:
                                menu_id = fn_rs_message['menuId']
                                rs = roledao.get_func_by_menu_id(menu_id)
                                if rs['data']:
                                    rs_menu.append(rs['data'])
                        rs['data'] = rs_menu
            cacher.RoleCache.set(key, rs, CacheDuration.Default)
        return rs

    def get(self, user_id, with_pwd=False):
        return self._get_by_id(user_id, with_pwd)

    def _generate_auth_token(self, user_id, pwd):
        """
        生成tokenKey
        :param user_id: 用户Id
        :param expiration: 过期时间
        :return: token数据
        """
        token = {
            'id': user_id,
            'key': pwd,
            'time': time.monotonic()}
        return base64.b64encode(pickle.dumps(token)).decode('ascii')

    def _generate_safe_token(self, user_id):
        s = Serializer(SysSettings.Token_Key, expires_in=1800)
        token = s.dumps({'id': user_id}).decode("ascii")
        return token

    def verify_safe_token(self, token):
        serializer = Serializer(SysSettings.Token_Key, expires_in=1800)
        token_info = serializer.loads(token)
        rs = get_result_model()
        rs['data'] = token_info
        return rs

    def verify_auth_token(self, token):
        """
        验证用户token是否有效
        :param token: 用户token
        :return: 成功返回用户数据
        """
        device_id = 0
        key = CacheKey.device_is_safe(device_id)
        is_safe = cacher.DeviceCache.get(key)
        if is_safe is None:
            is_safe = device_safe_config_read(SysSettings.Middleware_Safe_Config_Path)
        rs = self.get_login_user(token)
        if rs['status'] != 0:
            return rs
        login_info = rs['data']
        login_info['last_time'] = time.monotonic()
        login_info['system_last_time'] = time.time()
        login_info['ip'] =  rs['data']['ip']
        g.safe = is_safe
        password_key = ''
        try:
            if is_safe == SafeConfig.IsSafe:
                from H_9U.util import jwt_token
                rs = self.verify_safe_token(token)
                if rs['status'] != 0:
                    return rs
                user_id = rs['data']['id']
            else:
                token_info = pickle.loads(base64.b64decode(token.encode('ascii')))
                user_id = token_info['id']
                password_key = token_info.get('key', '')
        except Exception as e:
            logger.error(e)
            return get_result_model(ResInfo.User_Token_Not_Valid)
        # 登录正常
        rs = self.get(user_id, True)
        if rs['status'] != 0 or not rs['data']:
            rs = get_result_model(ResInfo.User_Not_Exist)
            return rs
        if is_safe == SafeConfig.IsNotSafe and rs['data']['password'] != password_key:
            return get_result_model(ResInfo.User_Token_Expired)
        if rs['data']['superadmin'] != 1:
            # permission_key = CacheKey.user_permissions(user_id)
            # permission_info = cacher.UserCache.get(permission_key)
            # permission_info = self.permission[user_id]
            # if permission_info is None:
            #     return get_result_model(ResInfo.User_Token_Expired)
            role_id, menu_id_list, input_id_list, output_id_list, screen_id_list, mvr_id_list = self.permission[user_id]
            des = self.verify_permissions(
                user_id,
                role_id,
                menu_id_list,
                input_id_list,
                output_id_list,
                screen_id_list,
                mvr_id_list)
            if des['status'] != 0:
                return des
        return rs

    def get_login_user(self, token):
        """根据token获取登录用户信息，"""
        rs = get_result_model()
        login_info = self.token_data.get(token, None)
        if not login_info:
            return get_result_model(ResInfo.User_Not_Login)

        if self.login_config['sessionExpireType'] == SecuritySessionExpiredType.UnOverdue:
            rs['data'] = login_info
            return rs

        if self.login_config['sessionExpireType'] == 1:
            opt_time = login_info['login_time']
        else:
            opt_time = login_info['last_time']
        timespan = time.monotonic() - opt_time
        if timespan > self.login_config['sessionExpireTime'] * 60:
            self.token_data.pop(token)
            return get_result_model(ResInfo.User_Token_Expired)
        rs = get_result_model()
        rs['data'] = login_info
        return rs

    def build_user_data(self, token):
        """
        验证用户token是否有效
        :param token: 用户token
        :return: 成功返回用户数据
        """
        device_id = 0
        key = CacheKey.device_is_safe(device_id)
        is_safe = cacher.DeviceCache.get(key)
        password_key = ''
        if is_safe is None:
            is_safe = device_safe_config_read(SysSettings.Middleware_Safe_Config_Path)
        if is_safe == SafeConfig.IsSafe:
            rs = self.verify_safe_token(token)
            if rs['status'] != 0:
                return rs
            user_id = rs['data']['id']
        else:
            token_info = pickle.loads(base64.b64decode(token.encode('ascii')))
            user_id = token_info['id']
            password_key = token_info.get('key', '')
        rs = self.get(user_id, True)
        role_id = 0
        from H_9U.service.role import rolesvc
        user_des = rolesvc.get_role_by_user_id(user_id)
        if user_des['status'] == 0 and user_des['data']:
            role_id = user_des['data']['roleId']
        if rs['status'] != 0 or not rs['data']:
            return get_result_model(ResInfo.User_Not_Exist)
        if is_safe == SafeConfig.IsNotSafe and rs['data']['password'] != password_key:
            return get_result_model(ResInfo.User_Token_Expired)
        role_rs, role_type = role_filter_data_func(role_id)
        if rs['status'] == 0 and role_rs['status'] == 0:
            rs['data']['roleId'] = role_id
            rs['data']['roleType'] = role_type
            rs['data']['filterData'] = role_rs['data']
        return rs

    def verify_permissions(
            self,
            user_id,
            role_id,
            menu_id_list,
            input_id_list,
            output_id_list,
            screen_id_list,
            mvr_id_list):
        """
        验证权限是否改变
         从 请求头permission中获取登录时用户拥有的权限集合; 然后查询数据库获取该用户最新的权限集合,两者比较, 有任何改变 则强制用户退出.重新登录
        :param user_id: user_id
        :param permission: permission
        :return:
        """
        # 获取该用户最新的权限集合
        role_id_new, menu_id_list_new, input_id_list_new, output_id_list_new, screen_id_list_new, mvr_id_list_new = self.get_permission_list(
            user_id)
        rs = get_result_model()
        if role_id != role_id_new:
            return get_result_model(ResInfo.User_Token_Expired)
        if compare_elements(menu_id_list, menu_id_list_new) is False:
            return get_result_model(ResInfo.User_Token_Expired)
        if compare_elements(input_id_list, input_id_list_new) is False:
            return get_result_model(ResInfo.User_Token_Expired)
        if compare_elements(output_id_list, output_id_list_new) is False:
            return get_result_model(ResInfo.User_Token_Expired)
        if compare_elements(screen_id_list, screen_id_list_new) is False:
            return get_result_model(ResInfo.User_Token_Expired)
        if compare_elements(mvr_id_list, mvr_id_list_new) is False:
            return get_result_model(ResInfo.User_Token_Expired)
        return rs

    def add(self, data, is_safe=None):
        """
        添加用户
        :param data: 用户数据
        :return: 结果对象
        """
        username = data['username']
        password = data['password']
        rs = self._get_by_username(username)
        if rs['status'] == 0 and rs['data']:
            return get_result_model(ResInfo.User_Name_Exist)
        if is_safe == SafeConfig.IsSafe:
            password = generate_password_hash(password, method="pbkdf2:sha256:10000", salt_length=16)
        if is_safe == SafeConfig.IsNotSafe:
            password = md5(password)
        tel = data.get('tel', '')
        if tel != "":
            result = regular_check_phone(tel)
            if result is False:
                return get_result_model(ResInfo.User_Phone_Format_Err)
        user_email = data.get('email', '')
        if user_email != "":
            result = regular_check_email(user_email)
            if result is False:
                return get_result_model(ResInfo.User_Email_Format_Err)
        status = data.get('status', 1)
        role_id = data.get('roleId')
        from H_9U.service.role import rolesvc
        role_rs = rolesvc.get_role_by_id(role_id)
        if role_rs['status'] == 0 and role_rs['data']:
            role_type = role_rs['data']['role_type']
            if role_type == 1:
                return get_result_model(ResInfo.User_Add_Role_Err)
        return self._insert(
            username,
            password,
            tel,
            user_email,
            status,
            role_id)

    def update(self, user_id, params):
        """
        根据用户Id更新用户
        :param user_id: user_id
        :param params: params
        :return:
        """
        user_params = dict()
        tel = params.get('tel', '')
        if tel != "":
            result = regular_check_phone(tel)
            if result is False:
                return get_result_model(ResInfo.User_Phone_Format_Err)
        user_email = params.get('email', '')
        if user_email != "":
            result = regular_check_email(user_email)
            if result is False:
                return get_result_model(ResInfo.User_Email_Format_Err)
        user_params['tel'] = tel
        user_params['email'] = user_email
        role_id = params.get('roleId')
        if 'status' in params:
            user_params['status'] = params.get('status')
        if role_id <= 0:
            return get_result_model(ResInfo.Params_Error)
        rs = self._update(user_id, role_id, user_params)
        if rs['status'] == 0:
            cacher.UserCache.delete(CacheKey.user_detail(user_id))
            cacher.UserCache.delete(CacheKey.user_func(user_id))
        return rs

    def update_password(self, user_id, password, active=None, is_safe=None):
        '''
        修改密码
        :param user_id: 用户Id
        :param password: 密码
        :return: 结果对象
        '''
        if is_safe == SafeConfig.IsSafe:
            password = generate_password_hash(password, method="pbkdf2:sha256:10000", salt_length=16)
        if is_safe == SafeConfig.IsNotSafe:
            password = md5(password)
        rs = self._update_password(user_id, password, active)
        return rs

    def get_all(self):
        """
        获取全部用户
        :return: 用户列表
        """
        user_list = []
        rs = get_result_model()
        user_rs = self._get_all_user()
        if user_rs['status'] == 0 and user_rs['data']:
            for user_message in user_rs['data']:
                user_id = user_message['id']
                role_rs = roledao.get_role_by_user_id(user_id)
                if role_rs['status'] == 0 and role_rs['data']:
                    role_id = role_rs['data']['roleId']
                    role_message = roledao.get_by_id(role_id)
                    if role_message['status'] == 0 and role_message['data']:
                        user_message['role'] = role_message['data']
                    user_list.append(user_message)
        rs['data'] = user_list
        return rs

    def logout(self, token):
        """
        token用户logout
        :param token: 用户token
        :return: 结果对象
        """
        if token in self.token_data.keys():
            self.token_data.pop(token)
        return get_result_model()

    def delete(self, user_id):
        """
        删除用户
        :return: 结果
        """
        user_rs = self._delete(user_id)
        if user_rs['status'] == 0 and user_rs['data']:
            is_admin = user_rs['data']['superadmin']
            if is_admin:
                return get_result_model(ResInfo.User_Func_Err)
        rs = userdao.delete(user_id)
        if rs['status'] == 0:
            cacher.UserCache.delete(CacheKey.user_detail(user_id))
            cacher.UserCache.delete(CacheKey.user_func(user_id))
        return rs

    def all_funcs(self):
        return self._get_all_func()

    def get_license_user(self):
        rs = userdao.get_license_user()
        return rs

    def get_user_by_role_id(self, role_id):
        """
        获取角色获取用户列表
        :return: 结果
        """
        rs = roledao.get_by_id(role_id)
        role_list = []
        id_roles = []
        role_list.append(rs['data'])
        if rs['status'] != 0 or not rs['data']:
            return get_result_model(ResInfo.User_Not_Exist)
        role_ids = rs['data']['id']
        id_roles.append(role_ids)
        length = len(rs['data'])
        while length:
            for ids in id_roles:
                rs = roledao.get_by_parent_id(ids)
                length = len(rs['data'])
                if rs['status'] == 0 and rs['data']:
                    for rs_one in rs['data']:
                        role_ids = rs_one['id']
                        id_roles.append(role_ids)
                        role_list.append(rs_one)
        user_list = []
        self.__build_role_data(role_list, rs, user_list)
        rs['data'] = user_list
        return rs

    def __build_role_data(self,role_list, rs, user_list):
        for roles in role_list:
            role_user_rs = roledao.get_user_by_role_id(roles['id'])
            if role_user_rs['status'] != 0 or not role_user_rs['data']:
                rs['data'] = user_list
                return rs
            for role_user in role_user_rs['data']:
                user_id = role_user['userId']
                user_rs = self._get_by_id(user_id)
                role_message = roledao.get_by_id(roles['id'])
                if role_message['status'] == 0 and role_message['data']:
                    user_rs['data']['role'] = role_message['data']
                if user_rs['status'] == 0 and user_rs['data']:
                    user_list.append(user_rs['data'])

    def detail(self, user_id):
        """
        根据用户Id获取用户详情
        :param user_id:
        :return:
        """
        key = CacheKey.user_detail(user_id)
        user_rs = cacher.UserCache.get(key, None)
        if not user_rs:
            user_rs = self._get_by_id(user_id)
            if user_rs['status'] != 0 or user_rs['data'] is None:
                return get_result_model(ResInfo.User_Not_Exist)
            user_role_rs = roledao.get_role_by_user_id(user_id)
            if user_role_rs['status'] == 0 and user_role_rs['data']:
                role_id = user_role_rs['data']['roleId']
                rs = roledao.get_by_id(role_id)
                role_list = []
                id_roles = []
                role_list.append(rs['data'])
                if rs['status'] == 0 and rs['data']:
                    role_ids = rs['data']['id']
                    id_roles.append(role_ids)
                    length = len(rs['data'])
                    while length:
                        for ids in id_roles:
                            rs = roledao.get_by_parent_id(ids)
                            length = len(rs['data'])
                            if rs['status'] == 0 and rs['data']:
                                for rs_one in rs['data']:
                                    role_ids = rs_one['id']
                                    id_roles.append(role_ids)
                                    role_list.append(rs_one)
                role_message_list = []
                for roles in role_list:
                    role_list_rs = roledao.get_by_id(roles['id'])
                    if role_list_rs['status'] == 0 and role_list_rs['data']:
                        role_message_list.append(role_list_rs['data'])
                user_rs['data']['roles'] = role_message_list
            else:
                return get_result_model(ResInfo.Role_Data_Err)
            cacher.UserCache.set(key, user_rs, CacheDuration.Default)
        return user_rs

    def __build_role_info_data(self, rs, role_list, role_message_list):
        id_roles = []
        role_ids = rs['data']['id']
        id_roles.append(role_ids)
        length = len(rs['data'])
        while length:
            for ids in id_roles:
                rs = roledao.get_by_parent_id(ids)
                length = len(rs['data'])
                if rs['status'] == 0 and rs['data']:
                    for rs_one in rs['data']:
                        role_ids = rs_one['id']
                        id_roles.append(role_ids)
                        role_list.append(rs_one)
        for roles in role_list:
            role_list_rs = roledao.get_by_id(roles['id'])
            if role_list_rs['status'] == 0 and role_list_rs['data']:
                role_message_list.append(role_list_rs['data'])

    def get_permission_list(self, user_id):
        """
        获取当前用户的角色与权限
        :param user_id: 用户Id
        :param expiration: 过期时间
        :return: token数据
        """
        role_rs = roledao.get_role_by_user_id(user_id)
        if role_rs['status'] != 0:
            return get_result_model(ResInfo.Role_Jurisdiction_Code)
        role_id = role_rs['data']['roleId']
        input_id_list = []
        output_id_list = []
        screen_id_list = []
        mvr_id_list = []
        menu_id_list = []
        # 获取菜单权限
        input_rs = roledao.get_input_by_role_id(role_id)
        if input_rs['status'] == 0 and input_rs['data']:
            input_id_list = [x['inputId'] for x in input_rs['data']]
        output_rs = roledao.get_output_by_role_id(role_id)
        if output_rs['status'] == 0 and output_rs['data']:
            output_id_list = [x['outputId'] for x in output_rs['data']]
        screen_rs = roledao.get_screen_by_role_id(role_id)
        if screen_rs['status'] == 0 and screen_rs['data']:
            screen_id_list = [x['screenId'] for x in screen_rs['data']]
        mvr_rs = roledao.get_mvr_by_role_id(role_id)
        if mvr_rs['status'] == 0 and mvr_rs['data']:
            mvr_id_list = [x['mvrId'] for x in mvr_rs['data']]
        menu_rs = roledao.get_menu_by_role_id(role_id)
        if menu_rs['status'] == 0 and menu_rs['data']:
            menu_id_list = [x['funcId'] for x in menu_rs['data']]
        return role_id, menu_id_list, input_id_list, output_id_list, screen_id_list, mvr_id_list

    def user_security_read(self, device_id):
        from H_9U.service.security import securitysvc
        pwd_rs = securitysvc.password_rule_read(device_id)
        login_rs = securitysvc.login_rule_read(device_id)
        if pwd_rs['status'] != 0 or login_rs['status'] != 0:
            return get_result_model(ResInfo.Middle_Data_Err)
        rs = get_result_model()
        rs['data'] = {
            'loginConfig': login_rs['data'],
            'pwdConfig': pwd_rs['data']
        }
        return rs

    def user_role_message(self, user_id):
        """
        获取用户的角色信息
        param: user_id
        :return: 无分组信息
        """
        role_rs = roledao.get_role_by_user_id(user_id)
        role_type = None
        role_id = None
        if role_rs['status'] == 0 and role_rs['data']:
            # 获取当前用户拥有的角色
            role_id = role_rs['data']['roleId']
            role_rs = roledao.get_by_id(role_id)
            if role_rs['status'] == 0 and role_rs['data']:
                role_type = role_rs['data']['role_type']
        return role_type, role_id

    def user_role_type(self, role_id):
        """
        获取用户的角色类型
        param: user_id
        :return: 无分组信息
        """
        role_type = None
        role_rs = roledao.get_by_id(role_id)
        if role_rs['status'] == 0 and role_rs['data']:
            role_type = role_rs['data']['role_type']
        return role_type

    def reset_admin_password(self, id_code, password):
        """重置管理员密码"""
        check_id_rs = auth_code_dao.get_auth_code(id_code)
        if check_id_rs['status'] != 0:
            return check_id_rs
        elif (int(time.time()) - check_id_rs['data']['createTime']) > Auth_Code_Expire_Time:
            return get_result_model(ResInfo.Auth_Code_Expired)
        rs = userdao.get_admin_user()
        is_safe = self.__get_safe_config()
        if rs['status'] == 0 and rs['data']:
            user_id = rs['data']['id']
            active = UserStatus.Active
            rs = self.update_password(user_id, password, active, is_safe)
            if rs['status'] == 0:
                cacher.UserCache.delete(CacheKey.user_detail(user_id))
                cacher.UserCache.delete(CacheKey.user_list())
        return rs

    def get_admin_user(self):
        return userdao.get_admin_user()

    def get_verify_code(self, verify_id):
        """
        生成验证码
        :param verify_id:客户端guid
        :return: 验证码图片
        """
        rs = get_result_model()
        size = (80, 24)
        img = draw_image(size, 4)
        if img is None:
            rs['status'] = ResInfo.Middle_Data_Err
        text, img_str = img
        cacher.VerifyCodeCache.set(
            verify_id, text, ttl=CacheDuration.Default)
        rs['data'] = {
            'img': img_str,
            'text': text
        }
        return rs

    def token_list(self, device_id):
        rs = get_result_model()
        data = []
        for k, v in self.token_data.items():
            data.append({
                "token": k,
                "userId": v['userId'],
                "username": v['username'],
                "ip": v["ip"],
                "loginTime": time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(v['system_login_time'])),
                'lastTime': time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(v['system_last_time']))
            })
        rs['data'] = data
        return rs

usersvc = UserSvc()
