"""
Created on 2019年6月28日
@author: 刘益宗
input接口.
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
import math
import os
import re
import time
import csv
import threading
import uuid
from copy import deepcopy

from H_9U.models.result import get_result_model, ResInfo
from H_9U.mao.ipc_mao import ipc_api
from H_9U.util.assembly_data import cutting_data
from H_9U.util.common import check_ip
from H_9U.util.fileopt import remove_file
from H_9U.service.common import upload_file
from H_9U.conf.syssettings import SysSettings
from H_9U.models.sysconst import IPCDefaultValue, IPCSlotId, IPCExportConst, IPCSourceType, UploadFileType, \
    IPCImportConst, IPCType, IPCCheckRowConst, IPCImportStatus, IPCTemplate, IpcVideoMode, IpcDecodeNumber, IPCCardType, \
    SourceType
from H_9U.util.log import logger
from H_9U.util.regular_check import regular_check_ipc_source_ip


class IPCSvc:
    def __init__(self):
        self.import_task = {}

    def ipc_stream_detail(self, device_id, data):
        """
        读取ipc码流详情
        :param device_id: 设备 Id
        :param data: 码流Idlist
        :return: 结果对象
        """
        rs = ipc_api.read_ipc_stream_detail(device_id, data)
        if rs['status'] == 0 and rs['data']:
           for stream in rs['data']['streamList']:
               self.ipc_slot_total_count(stream)
        return rs

    def ipc_slot_stream_list(self, device_id):
        """
         设备的码流详情列表数据
        :param device_id: 设备Id
        :return: 结果对象
        """
        rs = ipc_api.read_ipc_stream_list(device_id)
        if rs['status'] == 0 and rs['data'] and rs['data']['slotList']:
            for stream in rs['data']['slotList']:
                self.ipc_slot_total_count(stream)
        return rs

    def ipc_slot_total_count(self, stream):
        if stream:
            video_mode = stream['videoMode']
            hard_ware_type = stream['hardwareType']
            # videoMode = 1 能解 4路
            # videoMode = 2 能解 16路
            # videoMode = 3 能解 8路
            # videoMode = 4 能解 2路
            # videoMode = 0 拼接模式 海思能解1路 RK能解2路
            if video_mode == IpcVideoMode.FourWayMode:
                stream['slotTotalCount'] = IpcDecodeNumber.FourWayNumber
            elif video_mode == IpcVideoMode.EightWayMode:
                stream['slotTotalCount'] = IpcDecodeNumber.EightWayNumber
            elif video_mode == IpcVideoMode.SixteenWayMode:
                stream['slotTotalCount'] = IpcDecodeNumber.SixteenWayNumber
            elif video_mode == IpcVideoMode.SplicingMode:
                if hard_ware_type == IPCCardType.HS:
                     stream['slotTotalCount'] = IpcDecodeNumber.OneWayNumber
                if hard_ware_type == IPCCardType.RK:
                     stream['slotTotalCount'] = IpcDecodeNumber.TwoWayNumber
            elif video_mode == IpcVideoMode.TwoWayMode:
                if hard_ware_type == IPCCardType.HS:
                     stream['slotTotalCount'] = IpcDecodeNumber.OneWayNumber
                if hard_ware_type == IPCCardType.RK:
                     stream['slotTotalCount'] = IpcDecodeNumber.TwoWayNumber
            else:
                stream['slotTotalCount'] = 0

    def ipc_source_list(self, device_id, data):
        """
        读取信号源列表，data为分页数据
        :param device_id: 设备Id
        :param data: 分页参数
        :return 结果数据
        """
        return ipc_api.read_ipc_source_list(device_id, data)

    def ipc_source_free_list(self, device_id, data):
        """
        读取除分组外的信号源列表，data为分页数据
        :param device_id: 设备Id
        :param data: 分页参数
        :return 结果数据
        """
        # 1.先查询除当前分组有的所有源id
        id_list = self.get_source_id_list(device_id)
        index = 1
        size = 100
        source_list = []
        rs = get_result_model()
        while index > 0:
            data['seqPageIndex'] = index - 1
            data['seqPageSize'] = size
            rs = ipc_api.read_ipc_source_free_list(device_id, data)
            if rs['status'] == 0 and rs['data']:
                source_list.extend(rs['data']['sourceList'])
                nums = rs['data']['nums']
                if math.ceil(nums / size) > index:
                    index += 1
                else:
                    index = 0
            else:
                break
        # 2. 过滤掉分组内占用的源id
        if source_list:
            for index_c in range(len(source_list) - 1, -1, -1):
                source = source_list[index_c]
                source_id = source['sourceId']
                if source_id in id_list:
                    source_list.remove(source_list[index_c])
        rs['data']['sourceList'] = source_list
        return rs

    def get_source_id_list(self, device_id):
        """
        获取所有分组内占用的ipc源的id集合
        :param device_id: 设备Id
        :return: 结果数据
        """
        id_list = []
        rs = ipc_api.read_ipc_group_list(device_id)
        if rs['status'] == 0 and rs['data']:
            id_list = self._get_source_id_list(device_id, rs['data'], id_list)
        return id_list

    def _get_source_id_list(self, device_id, data, id_list):
        """
         递归获取所有分组下的源id 集合数据
         :param device_id: 设备Id
         :return: 结果数据
         """
        if 'groupList' in data and len(data['groupList']) > 0:
            for group in data['groupList']:
                group_id = group['groupId']
                group_node_rs = ipc_api.read_ipc_group_node_list_without_page(device_id, group_id)
                if group_node_rs['status'] == 0 and group_node_rs['data']:
                    for x in group_node_rs['data']['sourceList']:
                        id_list.append(x['sourceId'])
                self._get_source_id_list(device_id, group, id_list)
        return id_list

    def ipc_source_layout_list(self, device_id, channel_num, data):
        """
        读取信号源布局列表，data为分页数据
        :param device_id: 设备Id
        :param channel_num: 需要的通道数量
        :param data: 分页参数
        :return 结果数据
        """
        return ipc_api.ipc_source_layout_list(device_id, channel_num, data)

    def ipc_source_list_by_slot_id(self, device_id, slot_id):
        """
        通过soltId查询源
        :param device_id: 设备Id
        :param slot_id: slot_id
        :return 结果数据
        """
        return ipc_api.read_source_list_by_slot_id(device_id, slot_id)

    def ipc_channel_detail_list(self, device_id, source_id):
        """
        读取信号源通道列表
        :param device_id: 设备Id
        :param source_id: 源Id
        :return 结果数据
        """
        rs = ipc_api.ipc_channel_detail_list(device_id, source_id)
        return rs

    def ipc_top_channel(self, device_id, source_id):
        rs = ipc_api.read_ipc_channel_list(device_id, source_id)
        if rs['status'] == 0 and rs['data']:
            if rs['data']['channelList']:
                channel = rs['data']['channelList'][0]
                channel_id = channel['channelId']
                rs = ipc_api.read_ipc_channel_detail(device_id, source_id, channel_id)
            else:
                return get_result_model(ResInfo.Middle_Data_Err)
        return rs

    def ipc_channel_detail(self, device_id, source_id, channel_id):
        """
        读取信号源通道详情
        :param device_id: 设备Id
        :param source_id: 源Id
        :param channel_id: 通道Id
        :return 结果数据
        """
        return ipc_api.read_ipc_channel_detail(
            device_id, source_id, channel_id)

    def ipc_source_detail(self, device_id, source_id):
        """
        读取信号源列表，data为分页数据
        :param device_id: 设备Id
        :param source_id: 源Id
        :return 结果数据
        """
        rs = ipc_api.read_ipc_source_detail(device_id, source_id)
        return rs

    def ipc_channel_edit(self, device_id, params):
        """
        修改通道数据
        :param device_id: 设备Id
        :param params: params
        :return 结果数据
        """
        return ipc_api.write_ipc_channel_edit(device_id, params)

    def ipc_channel_add(self, device_id, data):
        """
        add通道数据
        :param device_id: 设备Id
        :param data: params
        :return 结果数据
        """
        channels = deepcopy(data['channelList'])
        index = 0
        while index < len(channels):
            data['channelList'] = channels[index:index + IPCDefaultValue.Batch_Create_Default_Count]
            rs = ipc_api.write_ipc_source_append(device_id, data)
            if rs['status'] != 0:
                return rs
            index += IPCDefaultValue.Batch_Create_Default_Count
        return get_result_model()

    def ipc_channel_delete(self, device_id, params):
        """
        删除通道数据
        :param device_id: 设备Id
        :param params: params
        :return 结果数据
        """
        return ipc_api.write_ipc_channel_delete(device_id, params)

    def ipc_source_crops_list(self, device_id, source_id):
        """
        查询源截取列表
        :param device_id: 设备Id
        :param source_id: 源Id
        :return 结果数据
        """
        return ipc_api.read_ipc_crop_list(device_id, source_id)

    def ipc_source_detail_list(self, device_id, slot_id):
        """
        读取ipc source 详情列表
        :param device_id: 设备id
        :param slot_id: slot id
        :return: ipc列表
        """
        rs = ipc_api.read_ipc_source_list(device_id, slot_id)
        if rs['status'] == 0:
            source_list = []
            for source in rs['data']['ipcSourceList']:
                source_id = source['ipcSourceId']
                source_info = self.ipc_source_detail(device_id, source_id)
                if source_info['status'] == 0 and source_info['data']:
                    source_list.append(source_info['data'])
            rs['data'] = {'slotId': slot_id, 'ipcSourceList': source_list}

        return rs

    def ipc_camera_list(self, device_id, decode_capacity):
        """
        ipc摄像头列表
        :param device_id: 设备id
        :param decode_capacity: 规格
        :return: 摄像头列表
        """
        rs = ipc_api.read_ipc_camera_list(device_id, decode_capacity)
        return rs

    def ipc_slot_source_count(self, device_id, slot_id):
        """
        ipc slot 源数量
        :param device_id: 设备Id
        :param slot_id:  槽位Id
        :return: 源数量
        """
        source_count_rs = ipc_api.read_ipc_slot_source_count(
            device_id, slot_id)
        if source_count_rs['status'] == 0:
            source_channels_count = 0
            source_list = source_count_rs['data']['sourceList']
            for source in source_list:
                source_channels_count += source['channelCount']
            # 剩余的通道数量
            source_count_rs['data']['slotMostChannelCount'] = IPCDefaultValue.Slot_Max_Channel_Count - \
                source_channels_count
        return source_count_rs

    def ipc_slot_source_list(self, device_id, slot_id):
        """
        槽位下ipc源列表
        :param device_id: 设备Id
        :param slot_id: slot id
        :return: 结果对象
        """
        rs = ipc_api.read_ipc_slot_source_list(device_id, slot_id)
        return rs

    def ipc_search_source_list(self, device_id, data):
        """
        ipc 源搜索结果列表
        :param device_id: 设备Id
        :param data: 搜索参数
        :return:
        """
        rs = ipc_api.read_ipc_search_source_list(device_id, data)
        return rs

    def ipc_source_create(self, device_id, data):
        """
        批量创建，创建ipc源
        :param device_id: 设备id
        :param slot_id: slotId
        :param data: 数据
        :return: 结果对象
        """
        # 组装检验ip
        rs = self._assembly_data(data)
        if rs['status'] != 0:
            return rs
        channel_list = rs['data']['channelList']
        end_ips = rs['data']['endIp']
        ip_list = rs['data']['ipList']
        source_ip = rs['data']['sourceIp']
        source_name = rs['data']['sourceName']
        count_number = 1
        ips = cutting_data(source_ip, end_ips, ip_list)
        datas = []
        if len(ips) == 0:
            return get_result_model(ResInfo.Params_Error)
        elif len(ips) == 1:
            if source_name == "":
                data['sourceName'] = ips[0]
            data['ip'] = ips[0]
            for channel in channel_list:
                stream_list = channel['streamList']
                for stream in stream_list:
                    source_type = stream['protocol']['type']
                    # onvif 自动扫描时 需要我们补IP, 所以此处需要特殊处理
                    if source_type == 2:
                        stream['protocol']['onvif']['onvifIp'] = ips[0]
            datas.append(data)
        else:
            for ip in ips:
                channel_list_copy = deepcopy(channel_list)
                if source_name == "":
                    data['sourceName'] = ip + "_" + str(count_number)
                else:
                    data['sourceName'] = source_name + "_" + str(count_number)
                for channel in channel_list_copy:
                    channel['createTime'] = str(time.time())
                    stream_list = channel['streamList']
                    for stream in stream_list:
                        source_type = stream['protocol']['type']
                        # onvif
                        if source_type == 2:
                            stream['protocol']['onvif']['onvifIp'] = ip
                        # rtsp manufacturer
                        elif source_type == 1:
                            if stream['protocol']['rtsp']['rtspPort'] == "" \
                                    and stream['protocol']['rtsp']['manufacturer'] == 255:
                                stream['protocol']['rtsp']['rtspUrl'] = "rtsp://" + ip + "/"
                            else:
                                url_data = stream['protocol']['rtsp']['rtspUrl'].split('//')[1].split(':')[1]
                            # rtsp://192.168.10:554/cam/realmonitor?channel=1&subtype=0
                                stream['protocol']['rtsp']['rtspUrl'] = "rtsp://" + ip + ":" + str(url_data)
                            stream['protocol']['rtsp']['rtspIp'] = ip
                        # gb28181
                        else:
                            stream['protocol']['gb28181']['platformIp'] = ip
                data['channelList'] = channel_list_copy
                data['ip'] = ip
                datas.append(deepcopy(data))
                count_number += 1
        if data['ipcType'] == IPCType.NVR:
            rs = self._nvr_batch_create(device_id, datas)
        else:
            rs = self._ipc_batch_create(device_id, datas)
        return rs

    def ipc_import_source(self, device_id):
        """
        异步导入数据源，由单独导入任务去执行导入
        :param device_id: 设备id
        :return: 导入任务信息
        """
        up_rs = upload_file(UploadFileType.IPC)
        # 判断文件上传是否成功
        if not up_rs["bln"]:
            return get_result_model(ResInfo.Upload_File_Fail)
        path_nm = up_rs["pathNm"]
        # 读取导入文件
        lines = self._ipc_import_file_read(path_nm)
        # 删除导入文件
        remove_file(path_nm)
        if not lines:
            return get_result_model(ResInfo.Input_IPC_Import_File_Fmt_Err)
        total = len(lines) - IPCImportConst.Header_Count
        # 导入文件没数据
        if total <= 0:
            return get_result_model(ResInfo.Input_IPC_Import_Data_Err)
        import_id = str(uuid.uuid1())
        self.import_task[import_id] = {
            "result": {
                "total": total,
                "failNum": 0,
                "successNum": 0,
                "failFilePath": "",
                "importStatus": IPCImportStatus.Importing
            },
            "cancel": False
        }
        from flask import g
        timer = threading.Timer(0.5, self._ipc_import_task, [device_id, lines, import_id, g.token, g.user])
        timer.start()
        rs = get_result_model()
        rs["data"]["importId"] = import_id
        return rs

    def ipc_import_progress(self, import_id):
        """
        查询导入进度
        :param import_id: 导入任务id
        :return: 导入进度
        """
        if import_id not in self.import_task:
            return get_result_model(ResInfo.Input_IPC_Import_Progress_Fail_Err)
        result = deepcopy(self.import_task[import_id]["result"])
        if result["importStatus"] != IPCImportStatus.Importing:
            self.import_task.pop(import_id)
        rs = get_result_model()
        rs["data"] = result
        return rs

    def ipc_cancel_import(self, import_id):
        """
        取消导入
        :param import_id: 导入任务id
        :return: 取消结果
        """
        if not self.import_task[import_id]:
            return get_result_model(ResInfo.Input_IPC_Import_Cancel_Fail_Err)
        self.import_task[import_id]["cancel"] = True
        return get_result_model()

    def _ipc_import_file_read(self, path_nm):
        """
        按不同编码读取导入文件
        :param path_nm: 文件路径
        :param encoding: 编码
        :return: 读取结果
        """

        # 多种编码方式读取导入文件
        for encoding in IPCImportConst.Encoding_List:
            try:
                with open(path_nm, 'r', encoding=encoding) as f:
                    lines = [line for line in f if line.strip()]
                break
            except Exception:
                logger.exception('使用%s编码读取IPC导入文件失败' % encoding)
                lines = []
        return lines

    def _ipc_import_task(self, device_id, lines, import_id, token, user):
        """
        导入任务，具体执行导入数据的方法
        :param device_id: 设备id
        :param lines: 导入数据
        :param import_id: 导入任务id
        :param token: 本次导入任务创建者token
        :param user: 导次导入任务创建者
        """
        fail_list = []
        # 导入任务全局变量
        task = self.import_task[import_id]
        # 保存遇到的NVR源信息，信号源名称+规格+ip都相同，视为同一个源来追加通道
        # k-信号源名称_规格_ip_厂商,v-{channelIndex, sourceId}
        nvr_dict = {}
        # 加入当前app作用域
        from H_9U import app
        from flask import g
        from H_9U.api.websender import push_data_org
        from H_9U.models.syncdataname import SyncDataName
        with app.app_context():
            g.token = token
            g.user = user
            # 跳过表头，csv每个换行符算做一行
            for index, line in enumerate(lines[IPCImportConst.Header_Count:], IPCImportConst.Header_Count):
                # 判断取消标志位
                if task["cancel"]:
                    self.import_task.pop(import_id)
                    push_data_org(
                        SyncDataName.Input_IPC_Source_Create, {
                            'deviceId': device_id})
                    return
                print("line===", line)
                pattern = re.compile(r'(?:^|,)(?:"([^"]*)"|([^",\s]*))')
                row = [m.group(1) or m.group(2) for m in pattern.finditer(line)]
                for i in range(len(row)):
                    row[i] = row[i].strip("\n").strip(" ")
                # 校验数据
                check_rs = self._ipc_import_check_row_data(row)
                if check_rs  != "":
                    row.append(check_rs)
                    fail_list.append(row)
                    task["result"]["failNum"] += 1
                    continue
                # 构建导入源数据信息
                source = self._ipc_import_build_source_info(row)
                if source['ipcType'] == IPCType.NVR:
                    # 组装nvr字典key：信号源名称_规格_ip_厂商
                    nvr_key = "_".join([source["sourceName"], str(source["decodeCapacity"]), source["ip"],
                                        str(source["channelList"][0]["streamList"][0]["protocol"]["rtsp"]["manufacturer"])])
                    # nvr需要判断是否第一条，第一条创建，后面追加通道信息
                    if nvr_key in nvr_dict:
                        source["channelList"][0]["channelIndex"] = nvr_dict[nvr_key]["channelIndex"]
                        source["sourceId"] = nvr_dict[nvr_key]["sourceId"]
                        rs = ipc_api.write_ipc_source_append(device_id, source)
                        if rs["status"] == 0:
                            nvr_dict[nvr_key]["channelIndex"] += 1
                    else:
                        rs = ipc_api.write_ipc_source_create(device_id, IPCSlotId.Invalid, source)
                        if rs["status"] == 0:
                            nvr_dict[nvr_key] = {
                                "sourceId": rs['data']['sourceId'],
                                "channelIndex": 2
                            }
                else:
                    rs = ipc_api.write_ipc_source_create(device_id, IPCSlotId.Invalid, source)
                if rs["status"] != 0:
                    row.append(IPCCheckRowConst.Create_Source_Err)
                    fail_list.append(row)
                    task["result"]["failNum"] += 1
                    continue
                task["result"]["successNum"] += 1
                if index % IPCDefaultValue.Batch_Create_Default_Count == 0:
                    time.sleep(1)
            push_data_org(
                SyncDataName.Input_IPC_Source_Create, {
                    'deviceId': device_id})
        if fail_list:
            from H_9U.api.device import read_export_device_language
            language_rs = read_export_device_language(device_id)
            language_mode = language_rs['data']['languageMode']
            self._ipc_import_build_fail_file(import_id, fail_list,language_mode)
            return
        task["result"]["importStatus"] = IPCImportStatus.All_Success

    def _ipc_import_build_source_info(self, row):
        """
        将导入数据构建为待创建源信息
        :param row: 数据行
        :return: 待创建源信息
        """
        decode_capacity = row[1].upper()
        ipc_type = row[2].upper()
        source_type = row[3].upper()
        ip = row[7]
        username = row[5]
        password = row[6]
        if source_type == "ONVIF":
            source = deepcopy(IPCImportConst.ONVIF)
            main = source["channelList"][0]["streamList"][0]["protocol"]["onvif"]
            sub = source["channelList"][0]["streamList"][1]["protocol"]["onvif"]
            main["onvifIp"] = ip
            sub["onvifIp"] = ip
            main["onvifUsrName"] = username
            sub["onvifUsrName"] = username
            main["onvifPassWord"] = password
            sub["onvifPassWord"] = password
        else:
            source = deepcopy(IPCImportConst.RTSP)
            main = source["channelList"][0]["streamList"][0]["protocol"]["rtsp"]
            sub = source["channelList"][0]["streamList"][1]["protocol"]["rtsp"]
            main["rtspIp"] = ip
            sub["rtspIp"] = ip
            main["rtspPort"] = int(row[8])
            sub["rtspPort"] = int(row[8])
            main["rtspUsrName"] = username
            sub["rtspUsrName"] = username
            main["rtspPassWord"] = password
            sub["rtspPassWord"] = password
            manufacturer = row[4]
            if manufacturer in IPCImportConst.Manufacturer:
                manufacturer = IPCImportConst.Manufacturer[manufacturer]
            else:
                manufacturer = 255
            main["manufacturer"] = manufacturer
            sub["manufacturer"] = manufacturer
            main["rtspUrl"] = row[10]
            sub["rtspUrl"] = row[11]
        source["sourceName"] = row[0]
        source["ip"] = ip
        source["decodeCapacity"] = IPCImportConst.Decode_Capacity[decode_capacity]
        source["ipcType"] = IPCImportConst.IPC_Type[ipc_type]
        source["channelList"][0]["channelName"] = row[9]
        source["createTime"] = str(time.time())
        return source

    def _ipc_import_build_fail_file(self, import_id, fail_list, language_mode):
        """
        构建导入失败文件
        :param import_id: 导入任务id
        :param fail_list: 失败数据
        """
        # 构建路径+文件名
        file_name = 'failFile_' + str(int(time.time() * 1000)) + '.csv'
        target_file = os.path.join(SysSettings.Download_IPC_File_Path, file_name)
        if not os.path.exists(SysSettings.Download_IPC_File_Path):
            os.makedirs(SysSettings.Download_IPC_File_Path, mode=0o777, exist_ok=True)
        # 写入文件
        with open(target_file, 'a', encoding='utf-8_sig', newline='') as f:
            f.writelines(IPCTemplate.template_dict.get(language_mode))
            w = csv.writer(f)
            w.writerows(fail_list)
        file_url = target_file[target_file.index('download'):]
        # 存入文件地址
        self.import_task[import_id]["result"]["failFilePath"] = file_url
        self.import_task[import_id]["result"]["importStatus"] = IPCImportStatus.Part_Success

    def _ipc_import_check_row_data(self, row):
        """
        校验导入源每行数据
        :param row: 数据行
        :return: 校验结果
        """
        # 取出要校验的数据，增加可读性
        source_name = row[0]
        decode_capacity = row[1].upper()
        ipc_type = row[2].upper()
        source_type = row[3].upper()
        manufacturer = row[4].upper()
        ip = row[7]
        port = row[8]
        channel_name = row[9]
        main_url = row[10]
        sub_url = row[11]
        if source_name == "" or len(source_name) > IPCCheckRowConst.Source_Name_Length:
            return IPCCheckRowConst.Source_Name_Err
        elif decode_capacity not in IPCImportConst.Decode_Capacity:
            return IPCCheckRowConst.Decode_Capacity_Err
        elif ipc_type not in IPCImportConst.IPC_Type:
            return IPCCheckRowConst.IPC_Type_Err
        elif source_type not in IPCImportConst.Source_Type:
            return IPCCheckRowConst.Source_Type_Err
        elif ip == "" or not check_ip(ip):
            return IPCCheckRowConst.IP_Err
        elif source_type == "RTSP":
            if manufacturer not in IPCImportConst.Manufacturer:
                return IPCCheckRowConst.Manufacturer_Err
            elif port == "" or not str(port).isdigit() or int(port) > 65535:
                return IPCCheckRowConst.Port_Err
            elif main_url == "":
                return IPCCheckRowConst.Main_URL_ERR
            elif sub_url == "":
                return IPCCheckRowConst.Sub_URL_ERR
        elif source_type == "ONVIF" and ipc_type != "IPC":
            return IPCCheckRowConst.IPC_Type_Err
        elif channel_name == "" or len(channel_name) > IPCCheckRowConst.Channel_Name_Length:
            return IPCCheckRowConst.Channel_Name_Err

        return IPCCheckRowConst.Success

    def ipc_export_source(self, device_id, data):
        """
        导出源方法
        :param device_id: 设备id
        :param data: 查询源信息条件
        :return: 导出文件的地址信息
        """
        # 构建导出路径+文件名
        file_name = 'export_' + str(int(time.time() * 1000)) + '.csv'
        target_file = os.path.join(SysSettings.Download_IPC_File_Path, file_name)
        if not os.path.exists(SysSettings.Download_IPC_File_Path):
            os.makedirs(SysSettings.Download_IPC_File_Path, mode=0o777, exist_ok=True)
        count = 0
        # 开始分批查询数据导出，保证至少查询一次
        while True:
            # 查询源详情+通道详情
            source_rs = self.ipc_source_channel_list(device_id, data)
            if source_rs['status'] != 0 or not source_rs['data']['sourceList']:
                remove_file(target_file)
                return get_result_model(ResInfo.Input_IPC_Export_Source_Err)
            source_list = source_rs['data']['sourceList']
            # 构建导出数据
            rows = self._ipc_export_build_rows(source_list, data['language'])
            if count == 0:
                with open(target_file, 'a', encoding='utf-8_sig', newline='') as f:
                    f.writelines(IPCTemplate.template_dict.get(data['language']))
                # 根据查询返回源数量数据，计算查询次数
                count = math.ceil(source_rs['data']['nums'] / data['seqPageSize'])
            # 写入导出文件
            with open(target_file, 'a', encoding='utf-8_sig', newline='') as f:
                w = csv.writer(f)
                w.writerows(rows)
            if count <= 1:
                break
            data['seqPageIndex'] += 1
            count -= 1
            time.sleep(0.5)

        file_url = target_file[target_file.index('download'):]
        rs = get_result_model()
        rs['data']['filePath'] = file_url
        rs['data']['deviceId'] = device_id
        return rs

    def _ipc_export_build_rows(self, source_list, language):
        """
        构建导出数据
        :param source_list: 源信息列表
        :param language: 语言
        :return: 导出数据列表
        """
        # 构建导出表格数据
        rows = []
        for source in source_list:
            source_name = source['sourceName']
            manufacturer = -1
            ip = source['ip']
            port = ""
            main_url = ""
            sub_url = ""
            decode_capacity = IPCExportConst.Decode_Capacity[source['decodeCapacity']]
            ipc_type = IPCExportConst.IPC_Type[source['ipcType']]
            # 遍历通道，每个通道占导出表格一行
            for channel in source['channelList']:
                main_stream = channel['streamList'][0]
                sub_stream = channel['streamList'][1]
                source_type = main_stream['protocol']['type']
                channel_name = channel['channelName']
                # 判断配置方式：1-RTSP、2-ONVIF、0-GB；GB不导出
                if source_type == IPCSourceType.GB28181:
                    break
                elif source_type == IPCSourceType.RTSP:
                    rtsp = main_stream['protocol']['rtsp']
                    manufacturer = rtsp['manufacturer']
                    port = rtsp['rtspPort']
                    username = rtsp['rtspUsrName']
                    password = rtsp['rtspPassWord']
                    main_url = rtsp['rtspUrl']
                    sub_url = sub_stream['protocol']['rtsp']['rtspUrl']
                else:
                    onvif = main_stream['protocol']['onvif']
                    username = onvif['onvifUsrName']
                    password = onvif['onvifPassWord']
                source_type = IPCExportConst.Source_Type[source_type]
                manufacturer = IPCExportConst.Manufacturer[manufacturer][language]
                row = [source_name, decode_capacity, ipc_type, source_type, manufacturer, username, password, ip, port,
                       channel_name, main_url, sub_url]
                rows.append(row)
        return rows

    def ipc_source_channel_list(self, device_id, data):
        """
        查询源列表带 通道+码流 信息
        :param device_id: 设备id
        :param data: 分页数据
        :return: 查询结果
        """
        # 分页查询源详情
        source_rs = ipc_api.read_ipc_source_list(device_id, data)
        if source_rs['status'] != 0:
            return source_rs
        for source in source_rs['data']['sourceList']:
            # 遍历源，查询通道详情
            rs = ipc_api.ipc_channel_detail_list(device_id, source["sourceId"])
            if rs['status'] != 0:
                return rs
            # 组装源+通道详情数据
            source['channelList'] = rs['data']['channelList']
        return source_rs

    def ipc_source_update(self, device_id, data):
        """
        编辑ipc源
        :param device_id: 设备id
        :param data: 数据
        :return: 结果对象
        """
        rs = ipc_api.ipc_source_update_Info(device_id, data)
        return rs

    def ipc_source_detail_info(self, device_id, source_id):
        """
        读取ipc源
        :param device_id: 设备id
        :param source_id: 设备id
        :param data: 数据
        :return: 结果对象
        """
        return ipc_api.ipc_source_detail_Info(device_id, source_id)

    def _assembly_data(self, data):
        """
        获取参数，
        :param data: 数据
        :return: 结果对象
        """
        rs = get_result_model()
        source_name = data.get('sourceName')
        ip_list = data.get('ipList')
        data['createTime'] = str(time.time())
        channel_list = data.get('channelList')
        end_ips = data.get('endIp')
        source_ip = data.get('ip')
        if source_ip != "" and not regular_check_ipc_source_ip(source_ip):
                return get_result_model(ResInfo.Input_IPC_Slot_Source_Add_ip_Err)
        if end_ips != "" and not regular_check_ipc_source_ip(end_ips):
                return get_result_model( ResInfo.Input_IPC_Slot_Source_Add_ip_Err)
        if len(ip_list) > 0:
            for ip_info in ip_list:
                if not regular_check_ipc_source_ip(ip_info):
                    return get_result_model(ResInfo.Input_IPC_Slot_Source_Add_ip_Err)
        rs['data']['channelList'] = channel_list
        rs['data']['endIp'] = end_ips
        rs['data']['ipList'] = ip_list
        rs['data']['sourceIp'] = source_ip
        rs['data']['sourceName'] = source_name
        return rs

    def _nvr_batch_create(self, device_id, datas):
        """
        NVR模式添加源
        :param device_id: 设备id
        :param datas: 数据
        :return: 结果对象
        """
        rs = get_result_model()
        for data in datas:
            index = IPCDefaultValue.Batch_Create_Default_Count
            # 复制一份channelList 数据
            channels = deepcopy(data['channelList'])
            # 取通道数据
            data['channelList'] = channels[:index]
            # 创建数据
            rs = ipc_api.write_ipc_source_create(device_id, IPCSlotId.Invalid, data)
            if rs['status'] == 0 and rs['data']:
                # 拿到创建之后返回的sourceId
                data['sourceId'] = rs['data']['sourceId']
                while index < len(channels):
                    data['channelList'] = channels[index:index + IPCDefaultValue.Batch_Create_Default_Count]
                    append_rs = ipc_api.write_ipc_source_append(device_id, data)
                    if append_rs['status'] != 0:
                        return get_result_model(
                            ResInfo.Input_IPC_Slot_Source_Append_Err)
                    index += IPCDefaultValue.Batch_Create_Default_Count
        return rs

    def _ipc_batch_create(self, device_id, datas):
        """
        IPC模式添加源
        :param device_id: 设备id
        :param datas: 数据
        :return: 结果对象
        """
        index = 0
        rs = get_result_model()
        while index < len(datas):
            # 批量创建源
            data = datas[index:index + IPCDefaultValue.Batch_Create_Default_Count]
            rs = ipc_api.write_ipc_source_create(device_id, IPCSlotId.Invalid, data)
            if rs['status'] != 0:
                return rs
            index += IPCDefaultValue.Batch_Create_Default_Count
        return rs

    def ipc_source_delete(self, device_id, data):
        """
        删除输入信号源
        :param device_id: 设备id
        :param data: data
        :return: 结果对象
        """
        # 获取拼接详情
        rs = ipc_api.write_ipc_source_delete(device_id, data)
        return rs

    def ipc_source_general(self, device_id, source_id, data):
        """
        修改ipc general 信息
        :param device_id: device_id
        :param source_id: source_id
        :param data: data 信息
        :return: 结果对象
        """
        return ipc_api.write_ipc_source_general(device_id, source_id, data)

    def ipc_crop_list(self, device_id, source_id, source_type=SourceType.IPCSource):
        """
        读取IPC截取列表
        :param device_id: 设备id
        :param source_id: 源id
        :return:
        """
        rs = ipc_api.read_ipc_crop_list(device_id, source_id, source_type)
        if rs['status'] == 0 and rs['data']:
            crops = []
            for c in rs['data']['crops']:
                crop_id = c['cropId']
                crop = self.ipc_crop_detail(device_id, source_id, crop_id)
                if crop['status'] == 0 and crop['data']:
                    crops.append(crop['data'])
            rs['data']['crops'] = crops
        return rs

    def ipc_crop_create(self, device_id, source_id, data):
        """
        创建IPC截取
        :param device_id: 设备id
        :param source_id: 源id
        :param data:
        :return:
        """
        return ipc_api.write_ipc_crop_create(device_id, source_id, data)

    def ipc_crop_delete(self, device_id, source_id, data):
        """
        删除IPC截取
        :param device_id: 设备id
        :param source_id: 源id
        :param data:
        :return:
        """
        return ipc_api.write_ipc_crop_delete(device_id, source_id, data)

    def ipc_crop_general(self, device_id,  data):
        """
        修改ipc截取
        :param device_id: 设备id
        :param source_id: 源id
        :param crop_id: 截取id
        :param data:
        :return:
        """
        return ipc_api.write_ipc_crop_general(
            device_id,  data)

    def ipc_crop_detail(self, device_id, source_id, crop_id):
        """
        读取IPC截取详情
        :param device_id: 设备id
        :param source_id: 源id
        :param crop_id: 截取id
        :return:
        """
        return ipc_api.read_ipc_crop_detail(device_id, source_id, crop_id)

    def ipc_group_list(self, device_id):
        """
        ipc分组列表
        :param device_id: 设备Id
        :return: 结果数据
        """
        rs = ipc_api.read_ipc_group_list(device_id)
        if rs['status'] == 0 and rs['data']:
            self._get_source_count(device_id, rs['data'])
        return rs

    def _get_source_count(self, device_id, data):
        """
         递归补充分组下的源数据数量
         :param device_id: 设备Id
         :return: 结果数据
         """
        if 'groupList' in data and len(data['groupList']) > 0:
            for group in data['groupList']:
                group_id = group['groupId']
                group_node_rs = ipc_api.read_ipc_group_node_list_without_page(device_id, group_id)
                if group_node_rs['status'] == 0 and group_node_rs['data']:
                   group['totalCount'] = len(group_node_rs['data']['sourceList'])
                self._get_source_count(device_id, group)


    def ipc_group_source_list(self, device_id):
        """
        ipc分组列表(包含数据源)
        :param device_id: 设备Id
        :return: 结果数据
        """
        rs = ipc_api.read_ipc_group_list(device_id)
        if rs['status'] == 0 and rs['data']:
            self._get_source(device_id, rs['data'])
        return rs

    def _get_source(self, device_id, data):
        """
         递归补充分组下的源数据
         :param device_id: 设备Id
         :return: 结果数据
         """
        if 'groupList' not in data or len(data['groupList']) <= 0:
            return
        for group in data['groupList']:
            group_id = group['groupId']
            group_node_rs = ipc_api.read_ipc_group_node_list_without_page(device_id, group_id)
            if group_node_rs['status'] == 0 and group_node_rs['data']:
                group['sourceList'] = group_node_rs['data']['sourceList']
                for source in group['sourceList']:
                    detail_rs = self.ipc_channel_detail_list(device_id, source['sourceId'])
                    if detail_rs['status'] == 0 and detail_rs['data']:
                        source['channelList'] = detail_rs['data']['channelList']
            self._get_source(device_id, group)

    def ipc_group_create(self, device_id, data):
        """创建ipc分组"""
        sources = data['sourceList']
        for x in sources:
            if x['groupId'] != -1:
                data_list = {"groupId": x['groupId'], "sourceList": [
                    {"sourceId": x['sourceId']}], "deviceId": 0}
                rs = self.ipc_group_node_delete(
                    device_id, data_list)
                if rs['status'] != 0:
                    return get_result_model(
                        ResInfo.Middle_Data_Err)
        return self.ipc_group_source_append(device_id, data)

    def ipc_group_source_append(self, device_id, data):
        """
        对分组源数据进行追加处理
        """
        datas = deepcopy(data)
        sources = datas['sourceList']
        datas['sourceList'] = sources[:IPCDefaultValue.Group_Max_Add_Source_Count]
        rs = ipc_api.ipc_group_create(device_id, datas)
        if rs['status'] == 0 and rs['data']:
            group_id = rs['data']['id']
            datas['groupId'] = group_id
        index = IPCDefaultValue.Group_Max_Add_Source_Count
        while index < len(sources):
            datas['sourceList'] = sources[index : IPCDefaultValue.Group_Max_Add_Source_Count + index]
            append_rs = ipc_api.ipc_group_node_add(device_id, datas)
            if append_rs['status'] != 0:
                return append_rs
            else:
                index += IPCDefaultValue.Group_Max_Add_Source_Count
        return rs

    def ipc_group_delete(self, device_id, data):
        """删除ipc分组"""
        return ipc_api.ipc_group_delete(device_id, data)

    def ipc_group_node_delete(self, device_id, data):
        """ipc分组删除节点"""
        return ipc_api.ipc_group_node_delete(device_id, data)

    def ipc_group_node_list_without_page(self, device_id, group_id):
        """获取ipc分组全部节点数据，不分页"""
        rs = ipc_api.read_ipc_group_node_list_without_page(device_id, group_id)
        if rs['status'] == 0 and rs['data']:
            for source in rs['data']['sourceList']:
                detail_rs = self.ipc_channel_detail_list(device_id, source['sourceId'])
                if detail_rs['status'] == 0 and detail_rs['data']:
                    source['channelList'] = detail_rs['data']['channelList']
        return rs

    def ipc_group_node_list(self, device_id, group_id):
        """ipc分组下源列表"""
        return ipc_api.read_ipc_group_node_list(device_id, group_id)

    def ipc_group_update(self, device_id, group_id, params):
        """更新ipc分组"""
        sources = params['sourceList']
        for x in sources:
            if x['groupId'] != -1 and x['groupId'] != group_id:
                data = {"groupId": x['groupId'], "sourceList": [
                    {"sourceId": x['sourceId']}], "deviceId": 0}
                rs = self.ipc_group_node_delete(device_id, data)
                if rs['status'] != 0:
                    return get_result_model(ResInfo.Middle_Data_Err)
        rs = ipc_api.ipc_group_clear(
            device_id, group_id, {
                'deviceId': device_id, "groupId": group_id})
        if rs['status'] == 0:
            rs = ipc_api.ipc_group_update(device_id, params)
        return rs

    def ipc_montage_list(self, device_id, slot_id):
        """ipc拼接源列表"""
        rs = ipc_api.ipc_montage_list(device_id, slot_id)
        return rs

    def ipc_montage_group_list(self, device_id, slot_id):
        """ipc拼接源列表"""
        rs = ipc_api.ipc_montage_group_list(device_id, slot_id)
        return rs

    def ipc_montage_template_create(self, device_id, data):
        """ipc 拼接源创建模板"""
        return ipc_api.ipc_montage_template_create(device_id, data)

    def ipc_montage_template_detail(self,device_id, template_id):
        """ipc 拼接源模板详情"""
        return ipc_api.ipc_montage_template_detail(device_id, template_id)

    def ipc_montage_template_update(self, device_id, data):
        """ipc 拼接源模板更新"""
        return ipc_api.ipc_montage_template_update(device_id, data)

    def ipc_montage_template_delete(self, device_id, data):
        """ipc 拼接源删除模板"""
        return ipc_api.ipc_montage_template_delete(device_id, data)

    def ipc_montage_template_apply(self, device_id, data):
        """ipc 拼接源应用模板"""
        return ipc_api.ipc_montage_template_apply(device_id, data)

    def ipc_montage_rename(self, device_id, data):
        """ipc 拼接源修改名称"""
        return ipc_api.ipc_montage_rename(device_id, data)
    # ipc 拼接结束

    def ipc_source_status_list(self, device_id, group_id, data):
        """
		ipc组内源信号的处理
		:param device_id: 设备Id
		:param group_id: group id
		:return: 结果对象
		"""
        return ipc_api.ipc_source_status_list(device_id, group_id, data)


    def write_ipc_bg_color(self, device_id, data):
        """
        设置IPC无源显示画面
        :param device_id: 设备id
        :param data: 设备参数
        :return: 结果对象
        """
        return ipc_api.write_ipc_bg_color(device_id, data)

    def read_ipc_bg_color(self, device_id):
        """
        查询IPC无源显示画面
        :param device_id: 设备id
        :return: 结果对象
        """
        return ipc_api.read_ipc_bg_color(device_id)

    def write_ipc_stream_rule(self, device_id, data):
        """
        设置IPC码流规则
        :param device_id: 设备id
        :param data: 码流规则参数
        :return: 结果对象
        """
        return ipc_api.write_ipc_stream_rule(device_id, data)

    def read_ipc_stream_rule(self, device_id):
        """
        查询IPC码流规则
        :param device_id: 设备id
        :return: 结果对象
        """
        return ipc_api.read_ipc_stream_rule(device_id)


    def ipc_source_list_and_first_channel(self, device_id, data):
        """
        查询源列表带 通道+码流 信息
        :param device_id: 设备id
        :param data: 分页数据
        :return: 查询结果
        """
        # 分页查询源详情
        source_rs = ipc_api.read_ipc_source_list(device_id, data)
        if source_rs['status'] == 0:
            for source in source_rs['data']['sourceList']:
                # 遍历源，查询通道详情
                rs = ipc_api.ipc_first_channel_detail(device_id, source["sourceId"])
                if rs['status'] != 0:
                    break
                # 组装源+通道详情数据
                first_channel= rs['data']['firstChannel']
                source_crop_rs = ipc_api.read_ipc_crop_list(device_id, first_channel['channelId'], source_type= SourceType.IPCSource)
                if source_crop_rs['status'] != 0 or not source_crop_rs['data']:
                    first_channel['cropList'] = []
                else:
                    first_channel['cropList'] = source_crop_rs['data']['cropList']
                source['firstChannel'] = rs['data']['firstChannel']
        return source_rs

    def ipc_montage_template_zorder_update(self, device_id, data):
        """ipc 拼接源模板更新"""
        return ipc_api.ipc_montage_template_zorder_update(device_id, data)

    def ipc_montage_group_update(self, device_id, data):
        """ipc 拼接源模板更新"""
        return ipc_api.ipc_montage_group_update(device_id, data)

    def ipc_montage_poll_status_write(self, device_id, data):
        """ipc 拼接源模板更新"""
        return ipc_api.ipc_montage_poll_status_write(device_id, data)

    # def read_ipc_crop_list(self, device_id, channel_id):
    #     """ipc 拼接源模板更新"""
    #     return ipc_api.read_ipc_crop_list(device_id, channel_id)



ipc_svc = IPCSvc()
