"""
Created on 2021年9月13日
@author: 程声清
output业务处理层.
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
import math

from H_9U.mao import device_mao
from H_9U.models.cachekey import CacheKey, CacheDuration
from H_9U.models.result import get_result_model, ResInfo
from H_9U.mao.output_mao import output_api
from H_9U.open.middle import open_device_middle
from H_9U.models.sysconst import ModelId, ConnectCapacity, InterfaceType, FunctionType, CardCategory, \
    TwoInOneCardData_1300, TwoInOneCardData_1040, CardType
from H_9U.service.common import get_connect_capacity
from H_9U.util.cache import cacher
from H_9U.util.log import logger
from H_9U.mao.impl import output
from H_9U.util.outputtiming import outputResolution, outputCustomResolution


def output_read_list_middle(device_id):
    """
    读取列表
    :param device_id: 设备Id
    :return: 结果对象
    """
    return output_api.read_list(device_id)


def output_original_detail_middle(device_id, output_id):
    """
    读取输出详细信息
    :param device_id: 设备Id
    :param output_id:  输出Id
    :return: 输出详细信息
    """
    return _output_read_detail_middle(device_id, output_id)


def _output_read_detail_middle(device_id, output_id):
    """
    读取详细信息
    :param device_id: 设备号
    :param output_id: 输入号
    :return: 结果对象
    """
    detail_rs = output_api.read_detail(device_id, output_id)
    if detail_rs["status"] == 0 and detail_rs['data']:
        detail_rs['data']['deviceId'] = device_id
        detail_rs['data']['connectCapacity'] = get_connect_capacity(
            detail_rs['data']['interfaceType'])
        if detail_rs['data']['modelId'] in ModelId.Two_In_One_Card_List:
            detail_rs['data']['connectCapacity'] = ConnectCapacity.K4
    return detail_rs


def write_timing_middle(device_id, output_id, time_ming):
    """
    设置Timing.
    :param device_id: 设备id
    :param output_id: 输入id.
    :param time_ming: 设置Timing.
    :return: 结果对象
    """
    # noinspection PyBroadException
    return output_api.write_timing(device_id, output_id, time_ming)


def write_4k_timing_middle(device_id, output_id, data):
    """
    设置Timing.
    :param device_id: 设备id
    :param output_id: 输入id.
    :param data: data
    :return: 结果对象
    """
    # noinspection PyBroadException
    return output_api.output_write_4k_timing(device_id, [output_id], data)


def output_write_payload_middle(device_id, output_list, params):
    """
    二合一卡设置带载面积并获取结果处理业务
    :param device_id: 设备id
    :param output_list: 输入列表.
    :param params: 二合一卡设置带载面积计算参数
    :return: 结果对象
    """
    return output_api.write_payload(device_id, output_list, params)


def two_in_one_outputs_middle(device_id):
    """
    返回设备二合一卡和对应的output
        {
            1:[1,2,3],
            2:[3,4,5]
        }
    :param device_id: 设备Id
    :return: 二合一卡信息
    """
    rs = _read_origin_list(device_id)
    two_in_one_rs = _read_tio_slots(device_id)

    if two_in_one_rs['status'] != 0:
        rs = get_result_model(ResInfo.Middle_Data_Err)
    elif rs['status'] == 0:
        slot_dict = {}
        # output存在，且存在二合一卡
        if rs['data']['outputs'] and two_in_one_rs['data']:
            two_in_one_list = two_in_one_rs['data']
            for item in rs['data']['outputs']:
                # 获取输入详情
                detail = output_read_origin_detail_middle(
                    device_id, item['outputId'])
                if detail['status'] == 0:
                    if detail['data']['slotId'] in two_in_one_list:
                        if detail['data']['slotId'] in slot_dict:
                            slot_dict[detail['data']['slotId']].append(
                                item['outputId'])
                        else:
                            slot_dict[detail['data']['slotId']] = [
                                item['outputId']]
                else:  # 读取输出详情失败，返回错误
                    return get_result_model(ResInfo.Middle_Data_Err)

        rs['data'] = slot_dict

    return rs


def _read_origin_list(device_id):
    """
    读取输出列表并加入缓存， 二合一卡显示四个接口
    :param device_id: 设备id
    :return: 输出列表
    """
    key = CacheKey.output_list(device_id)
    return cacher.OutputCache.try_get_value(
        key, CacheDuration.Default, output.output_list_read, device_id)


def _read_tio_slots(device_id):
    """
    读取二合一卡的slotId列表
    :param device_id:  设备Id
    :return: [slotId1,slotId2, ...]
    """
    return _two_in_one_cart_slots(device_id)


def _two_in_one_cart_slots(device_id):
    """
    获取设备二合一卡slot列表
    :param device_id:设备Id
    :return:[slotId1,slotId2, ...]
    """
    return device_mao.two_in_one_card_slot(device_id)


def output_read_origin_detail_middle(device_id, output_id):
    """
     读取output的原始信息
    :param device_id:
    :param output_id:
    :return:
    """
    key = CacheKey.output_detail(device_id, output_id)
    return cacher.OutputCache.try_get_value(
        key,
        CacheDuration.Default,
        output.output_full_detail_read,
        device_id,
        output_id)


def output_read_detail_middle(device_id, output_id):
    """
    读取输出详情并加入缓存
    :param device_id: 设备id
    :param output_id: 输出id
    :return: 输出详情
    """
    rs = output_read_origin_detail_middle(device_id, output_id)
    if rs['status'] == 0 and rs['data']:
        rs['data']['resolution'] = {
            'width': rs['data']['timing']['horizontal']['addrTime'],
            'height': rs['data']['timing']['vertical']['addrTime'],
            'refresh': rs['data']['timing']['refreshRate']
        }
        # 二合一卡处理
        if rs['data']['modelId'] in ModelId.Two_In_One_Card_List:
            slot_id = rs['data']['slotId']
            slot_rs = _read_tio_payload(device_id, slot_id)
            if slot_rs['status'] == 0:
                rs['data']['resolution'] = {
                    'width': slot_rs['data']['width'],
                    'height': slot_rs['data']['height'],
                    'refresh': slot_rs['data']['refresh'],
                    'direction': slot_rs['data']['direction'],
                    'outputCount': slot_rs['data']['outputCount']
                }

        # dl output 处理
        if rs['data']['modelId'] in ModelId.DL_Output_Card_List:
            dl_rs = _read_dl_outputs(device_id)
            if dl_rs['status'] == 0 and rs['data']['outputId'] in dl_rs['data']['dlList']:
                if rs['data']['modelId'] == ModelId.Output_4_HDMI_Card:
                    rs['data']['interfaceType'] = InterfaceType.HDMI14
                else:
                    rs['data']['interfaceType'] = InterfaceType.Dual_Link_DVI

                rs['data']['resolution']['width'] *= 2
                rs['data']['timing']['horizontal']['addrTime'] *= 2
                rs['data']['timing']['horizontal']['frontPorch'] *= 2
                rs['data']['timing']['horizontal']['sync'] *= 2
                rs['data']['timing']['horizontal']['totalTime'] *= 2

        # 4K卡处理
        if rs['data']['modelId'] == ModelId.Output_2_HDMI20_Card:
            slot_id = rs['data']['slotId']
            slot_rs = _read_tio_payload(device_id, slot_id)
            output_count = 4
            if slot_rs['status'] == 0:
                output_count = slot_rs['data']['outputCount']
                rs['data']['resolution'] = {
                    'width': slot_rs['data']['width'],
                    'height': slot_rs['data']['height'],
                    'refresh': slot_rs['data']['refresh'],
                    'direction': slot_rs['data']['direction'],
                    'outputCount': slot_rs['data']['outputCount']
                }
            if output_count == 4:
                rs['data']['timing']['horizontal']['addrTime'] *= 4
                rs['data']['timing']['horizontal']['frontPorch'] *= 4
                rs['data']['timing']['horizontal']['sync'] *= 4
                rs['data']['timing']['horizontal']['totalTime'] *= 4
    return rs


def _read_tio_payload(device_id, slot_id):
    """
    读取二合一卡的带载信息
    :param device_id: 设备Id
    :param slot_id: slotId
    :return: {width, height, refreshRate}
    """
    return device_mao.read_two_in_one_card_payload(device_id, slot_id)


def _read_dl_outputs(device_id):
    """
    读取所有dl的output
    :param device_id:
    :return:
    """
    rs = device_mao.output_dl_slot(device_id)
    if rs['status'] != 0:
        logger.error('读取DL output slot err code:' + str(rs['status']))
        return rs

    dis_list = []
    dl_list = []
    if rs['data']:
        o_rs = _read_origin_list(device_id)
        if o_rs['status'] != 0:
            logger.error('读取中间件output_list err code:' + str(o_rs['status']))
            return o_rs

        slots = rs['data']
        for item in o_rs['data']['outputs']:
            # 获取输入详情
            detail = output_read_origin_detail_middle(
                device_id, item['outputId'])
            if detail['status'] == 0:
                slot = [x for x in slots if x['slotId']
                        == detail['data']['slotId']]
                if slot:
                    slot = slot[0]
                    interface = [x for x in slot['interfaces']
                                 if x['interfaceId'] == detail['data']['interfaceId']][0]
                    if interface['functionType'] == FunctionType.Disable:
                        dis_list.append(item['outputId'])
                    elif interface['functionType'] == FunctionType.DL:
                        dl_list.append(item['outputId'])
            else:  # 读取输出详情失败，返回错误
                logger.error('读取output详情错误：' + str(detail['status']))
                return get_result_model(ResInfo.Middle_Data_Err)

    rs['data'] = {
        'disList': dis_list,
        'dlList': dl_list
    }
    return rs


def get_tio_output_timing_middle(
        card_category,
        full_width,
        full_height,
        output_refresh):
    """
      :param card_category
      :param full_width
      :param full_height
      :param output_refresh
           1.是否超出最大带载 数据限制：
               是否超出最大带载？1041W*60 / 1312W*60
               否：是否满足宽高限制 宽（最大/最小）：2560*4/800 高：8192/600
                   否：不能带载
                   是：是否满足时钟频率限制：H_total*V_total*refresh <165M
                       否：不能带载
                       是：正常逻辑

           2. 设置二合一卡的负载信息
           屏幕是否大于1920*1080*60 ？
               是：
                   水平是否大于1920？
                       是：水平四分割 (水平建屏）(优先CVT规则，满足不了的情况下，采用减少水平逆程的规则）
                       否：屏幕是否大于1920*1080*60*3 ？
                           是：  使用实际水平大小向上对2取整且垂直建屏并做截取，单接口可支持垂直8192(按照CVT规则计算，一个接口不满足，采用两个接口建屏，\
                           依次类推，在4个接口垂直建屏的情况下，如果CVT规则满足不了，尝试采用减少水平逆程规则）（垂直建屏）
                           否：  水平四分割 (水平建屏）(优先CVT规则，满足不了的情况下，采用减少水平逆程的规则）
               否：
                   水平是否大于1920 ？
                       是：使用多个水平2560并做截取（水平建屏）
                       否：h_total*v_total*refresh <165M ?
                           是 ：    使用实际大小向上对2取整建屏
                           否  ： （垂直建屏）
                               使用实际水平大小向上对2取整且垂直建屏并做截取，单接口可支持垂直8192(按照CVT规则计算，一个接口不满足，采用两个接口建屏，依次类推）


           :param full_width: 二合一卡总体宽度
           :param full_height: 二合一卡总体高度
           :param output_refresh: 二合一卡刷新率
           :return: 结果对象
           """
    if card_category == CardCategory.TwoInOneCard_20:
        two_in_one_card_data = TwoInOneCardData_1300
    else:
        two_in_one_card_data = TwoInOneCardData_1040

    standard_timing = None
    direction = 0
    full_area = output_refresh * full_width * full_height
    timing_height = full_height
    # 判断是否在带载范围内
    if full_area > two_in_one_card_data.SlotMaxArea \
            or full_width > two_in_one_card_data.SlotMaxWidth \
            or full_height > two_in_one_card_data.SlotMaxHeight \
            or full_width < two_in_one_card_data.SlotMinWidth \
            or full_height < two_in_one_card_data.SlotMinHeight:
        return get_result_model(ResInfo.Output_Max_Payload_Exceed)

    # 大于1920*1080*60， 一个放不下
    if full_area > two_in_one_card_data.SingleOutputDefaultResolution:
        #  # 水平大于1920 或 总体小于1920*1080*60*3
        if full_width > two_in_one_card_data.SingleOutputStandardWidth \
                or full_area <= two_in_one_card_data.SingleOutputDefaultResolution * 3:
            # 水平四分割 (水平建屏）(优先CVT规则，满足不了的情况下，采用减少水平逆程的规则）
            new_width = math.ceil(math.ceil(full_width / 4) / 4) * 4
            # 修改分辨率值
            timing_width = new_width
        # 否
        else:
            direction = 1
            timing_width = math.ceil(full_width / 2) * 2
            # 单接口可支持垂直8192（按照cvt规则计算，一个接口不满足采用2-4个接口，cvt满足不了尝试减少水平逆程的规则）
            for x in range(1, 5):
                timing_height = math.ceil(full_height / x)
                standard_timing = outputResolution(
                    timing_width,
                    timing_height,
                    output_refresh,
                    two_in_one_card_data.TimingMaxBandWidth_1)
                if standard_timing:
                    break
            # cvt算法不能满足，则使用自定义算法
            if not standard_timing:
                timing_height = math.ceil(full_height / 4)
                standard_timing = outputCustomResolution(
                    timing_width,
                    timing_height,
                    output_refresh,
                    two_in_one_card_data.TimingMaxBandWidth_1)
                if not standard_timing:  # 不能带载，直接返回
                    return get_result_model(ResInfo.Output_Max_Payload_Exceed)

    # 小于1920*1080*60，一个放得下
    # 宽度大于单接口最大宽度2560，使用多个水平2560并做截取（水平建屏）
    elif full_width > two_in_one_card_data.SingleOutputMaxWidth:
        timing_width = two_in_one_card_data.SingleOutputMaxWidth
        timing_height = full_height
    # 一个放的下，宽度小于2560
    else:
        timing_width = math.ceil(full_width / 2) * 2
        timing_height = full_height
        standard_timing = outputResolution(
            timing_width,
            timing_height,
            output_refresh,
            two_in_one_card_data.TimingMaxBandWidth_2)

        # h_total*v_total*refresh >165M
        # 计算timing，如果timing为空
        if not standard_timing:  # 垂直建屏
            direction = 1
            for x in range(2, 5):
                timing_height = math.ceil(full_height / x)
                standard_timing = outputResolution(
                    timing_width,
                    timing_height,
                    output_refresh,
                    two_in_one_card_data.TimingMaxBandWidth_2)
                if standard_timing:
                    break
            if not standard_timing:  # 不能带载，直接返回
                return get_result_model(ResInfo.Output_Max_Payload_Exceed)

    # 新的算法计算Timing
    if not standard_timing:
        standard_timing = outputResolution(
            timing_width,
            timing_height,
            output_refresh,
            two_in_one_card_data.TimingMaxBandWidth_1)
    if not standard_timing:
        standard_timing = outputCustomResolution(
            timing_width,
            timing_height,
            output_refresh,
            two_in_one_card_data.TimingMaxBandWidth_1)
    if standard_timing:
        rs = get_result_model()
        standard_timing['direction'] = direction
        rs['data'] = standard_timing
        return rs

    return get_result_model(ResInfo.Output_Max_Payload_Exceed)


# def _get_4k_output_timing(width, height, refresh, timing):
#     if timing:
#         standard_timing = timing
#     else:
#         standard_timing = outputResolution(width, height, refresh, Output4KCardData.TimingMaxBandWidth)
#
#     rs = get_result_model()
#     if not standard_timing:
#         rs['status'] = ResInfo.Output_Max_Payload_Exceed
#     elif (float(standard_timing['horizontal']["totalTime"])
#           * float(standard_timing['vertical']["totalTime"])
#           * float(standard_timing['refreshRate']) > Output4KCardData.TimingMaxBandWidth):
#         rs['status'] = ResInfo.Output_Max_Payload_Exceed
#     else:
#         rs['data'] = standard_timing
#     return rs

# def _get_two_in_one_card_sorted_output(output_id, tio_outputs_dict):
#     """
#     判断output是否是二合一卡的output
#     :param outputId: outputId
#     :param tio_outputs_dict: 二合一卡 {slotId:[output1, output2, ...]}
#     :return: (slotId, sorted(output_list))
#     """
#     for key, value in tio_outputs_dict.items():
#         # 如果能找到interface的output，说明是二合一卡interface
#         if output_id in value:
#             value.sort()
#             return key, value

# def _write_single_tio_payload(device_id, slot_id, outputs, timing, resolution):
#     """
#     设置单个二合一卡的带载信息
#     :param device_id:
#     :param slot_id:
#     :param outputs:
#     :param timing:
#     :return:
#     """
#     timing['deviceId'] = device_id
#     partial = False
#     for o_id in outputs:
#         timing['outputId'] = o_id
#         rs = write_timing_middle(device_id, o_id, deepcopy(timing))
#         if rs['status'] != 0:
#             partial = True
#
#     # 修改slot中的宽高
#     rs = _write_tio_payload(device_id, slot_id, resolution)
#     if rs['status'] == 0 and partial:
#         rs = get_result_model(ResInfo.Output_Write_Timing_List_Error)
#     return rs

# def _write_tio_payload(device_id, slot_id, payload):
#     """
#     设置二合一的带载信息
#     :param device_id: 设备Id
#     :param slot_id:  slotId
#     :param payload: 带载信息
#     :return: 结果对象
#     """
#     return WriteTwoInOneCardPayload(device_id, slot_id, payload)


# def WriteTwoInOneCardPayload(device_id, slot_id, payload):
#     """
#     设置二合一卡的带载面积
#     :param device_id: 设备Id
#     :param slot_id: slot Id
#     :param payload: 带载面积对象
#     :return: 结果对象
#     """
#     data = {
#         'deviceId': device_id,
#         'slotId': slot_id,
#         'resolution': {
#             'width': payload['width'],
#             'height': payload['height'],
#             'refresh': payload['refreshRate'],
#             'direction': payload.get('direction', 0),
#             'outputCount': payload.get('outputCount', 4)
#         }
#     }
#     return open_device_middle.write_slot_middle(device_id, slot_id, data)

def output_write_image_quality_middle(device_id, output_id, params):
    """
    设置画质
    :param device_id: 设备Id
    :param output_id:  输出Id
    :param params:
    :return:  结果对象
    """
    return output_api.write_image_quality(device_id, output_id, params)


def output_write_gamma_middle(device_id, output_id, params):
    """
    output设置gamma数据
    :param device_id: 设备id
    :param output_id: output id
    :param params:  gamma数据
    :return: 结果对象
    """
    return output_api.write_gamma(device_id, output_id, params)


def output_write_fixed_position_middle(device_id, params):
    """
    设置输出定位
    :param device_id: 设备Id
    :param params: 输出定位参数
    :return: 结果对象
    """
    output_id = 255
    params['outputId'] = output_id
    return output_api.write_fixed_position(device_id, output_id, params)

# def same_card_tio_outputs_middle(device_id, output_id):
#     """
#     相同二合一卡的所有outputIdList
#     :param device_id: 设备Id
#     :param output_id: outputId
#     :return: output对应的二合一卡信息(slotId, [output1, output2, ...])，如果不是二合一卡返回None,
#     """
#     rs = get_result_model()
#     rs['data'] = None
#     tio_rs = _two_in_one_outputs_middle(device_id)
#     if tio_rs['status'] == 0 and tio_rs['data']:
#         tio_rs = _get_two_in_one_card_sorted_output(output_id, tio_rs['data'])
#         if tio_rs:
#             rs['data'] = tio_rs
#
#     return rs

# def same_card_k4_outputs_middle(device_id, output_id):
#     """
#     相同4k的所有outputIdList
#     :param device_id: 设备Id
#     :param output_id: outputId
#     :return: output对应的二合一卡信息(slotId, [output1, output2, ...])，如果不是4k返回None,
#     """
#     rs = get_result_model()
#     rs['data'] = None
#     k4_rs = _read_4k_outputs_middle(device_id)
#     if k4_rs['status'] == 0 and k4_rs['data']:
#         for key, value in k4_rs['data'].items():
#             # 如果能找到interface的output，说明是二合一卡interface
#             if output_id in value:
#                 value.sort()
#                 rs['data'] = (key, value)
#     return rs


def _read_4k_outputs_middle(device_id):
    """
    返回4k output卡和对应的output
        {
            1:[1,2,3],
            2:[3,4,5]
        }
    :param device_id: 设备Id
    :return: 4k卡信息
    """
    rs = _read_origin_list(device_id)
    slot_rs = _read_4k_slots(device_id)
    if slot_rs['status'] != 0:
        rs = get_result_model(ResInfo.Middle_Data_Err)
    elif rs['status'] == 0:
        slot_dict = {}
        # output存在，且存在4k
        if rs['data']['outputs'] and slot_rs['data']:
            slots = slot_rs['data']
            for item in rs['data']['outputs']:
                # 获取输入详情
                detail = output_read_origin_detail_middle(
                    device_id, item['outputId'])
                if detail['status'] == 0:
                    if detail['data']['slotId'] in slots:
                        if detail['data']['slotId'] in slot_dict:
                            slot_dict[detail['data']['slotId']].append(
                                item['outputId'])
                        else:
                            slot_dict[detail['data']['slotId']] = [
                                item['outputId']]
                else:  # 读取输出详情失败，返回错误
                    return get_result_model(ResInfo.Middle_Data_Err)

        rs['data'] = slot_dict
    return rs


def _read_4k_slots(device_id):
    """
    读取4k槽位数据
    :param device_id: 设备Id
    :return:
    """
    return output4k_slot_middle(device_id)


def output4k_slot_middle(device_id):
    """
    读取输出4k槽位数据
    :param device_id: 设备Id
    :return:
    """
    rs = open_device_middle.device_read_detail_middle(device_id)
    if rs['status'] == 0:
        slots = rs['data']['slotList']
        data = [x['slotId'] for x in slots
                if x['cardType'] == CardType.Out
                and x['modelId'] == ModelId.Output_2_HDMI20_Card]
        rs['data'] = data
    return rs


def output_rename_middle(device_id, output_id, data):
    """
    设置基本信息
    :param device_id: 设备id
    :param output_id: 输出id
    :param data: data 数据
    :return: 结果数据对象
    """
    return output_api.write_general(device_id, output_id, data)
