﻿"""
Created on 2019年6月28日
@author: 刘益宗
系统常量配置
"""
import enum


class UpgradeType:
    Full = 0
    Part = 1

    All_List = [Part, Full]


class ModelConfKey:
    """
    模块配置key，与数据库modelConf表配合使用
    """
    LanguageMode = 'LanguageMode'  # 语言模式
    ScreenLock = 'ScreenLock'  # 屏幕锁定
    LayerLock = 'LayerLock'  # 图层锁定
    RealtimeDisplay = 'RealtimeDisplay'  # 是否实时上屏
    PresetPlay = 'PresetPlay'
    VideoStatus = 'VideoStatus'
    NeutralMode = 'NeutralMode'  # 中性模式
    PresetShowId = 'PresetShowId'  # 是否展示场景Id


class UploadFileType:
    """
    上传文件类型
    """
    BKG = 1
    OSD = 2
    Device_Settings = 3
    Upgrade = 4
    Input_EDID = 5
    NomarkMode = 6
    Normal = 7
    Font = 8
    IPC = 9
    Sdp = 10


class LanguageMode:
    """
    语言模式
    """
    Default = 2 #前端跳转到语言选择界面
    English = 0
    Chinese = 1
    Tc = 3 #繁体
    Korean = 4 #韩语

    English_Name = "en"
    Chinese_Name = "zh"
    Tc_Name = "tc"
    Korean_Name = "ko"

    LanguageMode_Dict = {
        English : English_Name,
        Chinese : Chinese_Name,
        Tc: Tc_Name,
        Korean: Korean_Name
    }

    Language_Dict = {
        English_Name: English,
        Chinese_Name: Chinese,
        Tc_Name: Tc,
        Korean_Name: Korean
    }


class BKGFile:
    """
    bkg图片文件
    """
    # 最大图片数量
    Max_BKG_Count = 16

    # 图片大小限制
    bkg_limit = [
        {'width': 8192, 'height': 1024},
        {'width': 4096, 'height': 2048}
    ]

    def __init__(self, width, height):
        self.width = width
        self.height = height

    def valid_pic(self):
        """
        判断图片是否合法
        :return: True/False
        """
        for limit in self.bkg_limit:
            if self.width <= limit['width'] and self.height <= limit['height']:
                return True
        return False


class InterfaceType:
    """
    interfaceType 对应值
    """
    EXP = 1
    Single_Link_DVI = 2
    Dual_Link_DVI = 3
    HDMI13 = 4
    HDMI14 = 5
    HDMI20 = 6
    DP11 = 7
    DP12 = 8
    SDI_3G = 9
    VGA = 10
    CVBS = 11
    YpbPr = 12
    RJ45 = 13
    USB = 14
    HDBaseT = 15
    HDBaseT_4K = 16
    OpticalFiber = 17
    SDI_12G = 18
    DB9_MALE = 19
    DB9_FEMALE = 20
    IO = 21
    RELAY = 22
    IR = 23
    Phoenix = 24
    SDI_3G_OUT = 25
    RJ45DL = 26
    Optical2x_Fiber = 27
    HDMI21 = 31
    DP14 = 32


    # 允许设置EDID接口列表
    _EDID_Setting_List = [Single_Link_DVI, Dual_Link_DVI, HDMI13, HDMI14, HDMI20, DP11, DP12, OpticalFiber, HDBaseT, DB9_MALE, SDI_3G_OUT, RJ45, RJ45DL, SDI_12G, HDMI21, DP14]
    # 允许功能切换的接口列表
    _Allow_Function_Change_List = [Single_Link_DVI, Dual_Link_DVI, HDMI13, HDMI14, HDMI20, DP12, HDBaseT, DB9_MALE, DB9_FEMALE, RJ45, RJ45DL,HDMI21, DP14]

    @staticmethod
    def allow_edid_setting(interface_type, additional_param=None):
        """
        是否允许设置EDID
        :param interface_type: 接口类型
        :return:  1：允许，0: 不允许
        """
        if interface_type in InterfaceType._EDID_Setting_List:
            if additional_param is None:
                return 1
            else:
                return 0
        return 0


class CardType:
    """
    卡类型
    """
    NoCard = 0
    In = 1
    Out = 2
    Sender = 3
    Mvr = 4
    STD_IO_Card = 5


class ConnectCapacity:
    """
    链接容量，默认使用SL
    """
    SL = 0
    DL = 1
    K4 = 2

    Default = SL


class LayerType:
    """
    开窗时图层类型图层类型
    """
    # ,0：mix，1：SL , F8Plus使用1
    mix = 0
    SL = 1
    Default = SL


class SourceType:
    NoSource = 0  # 无源,
    InputSource = 1  # 输入类型,
    IPCSource = 3  # IPC类型
    Color = 4  # 纯色
    IPCMontage = 5  # IPC拼接源
    InputSourcePull = 6  #输入源的轮巡源
    IPC = 255
    NdiSource = 8 # Ndi源
    ST2110Source = 10 # 2110源

class IpcCropType:
    ReadCropType = 0
    WriteCropType = 1


class DeviceType:
    Unknown = 0
    ES = 1
    HS = 2
    H_9U = 0x750A  # H_9U
    H_5U = 0x750B  # H_Ctrl_5U
    H_2U = 0x750D  # H_Ctrl_2U
    H_15U = 0x750E  # H_Ctrl_15U
    H_15U_Pro = 0x7510
    H_20U = 0x751E # H_Ctrl_20U

    P_H_9U = 0x7525  # H_9U
    P_H_5U = 0x7524  # H_Ctrl_5U
    P_H_2U = 0x7523  # H_Ctrl_2U
    P_H_15U = 0x752C  # H_Ctrl_15U
    P_H_15U_PRO = 0x752D  # H_Ctrl_15UPRO

    # E系列
    Es_3 = 0x750F
    Es_7 = 0x7511

    Alita_9 = 0x751C
    Alita_15 = 0x751D
    Alita_2 = 0x7521
    Alita_5 = 0x7522
    Alita_20 = 0x7530


    ES_ALL = [Es_3, Es_7]
    HS_ALL = [H_9U, H_5U, H_2U, H_15U, H_15U_Pro, H_20U]
    Alita_ALL = [Alita_9, Alita_15, Alita_2, Alita_5, Alita_20]
    P_Main_All = [P_H_9U, P_H_5U, P_H_2U, P_H_15U, P_H_15U_PRO]


class UpgradeConfig:
    H_160T_HardVer = '3.0'
    H_160T_FPGA_Filename = 'FPGA_160T.dat'
    H_RK_MCU_Filename = 'MCU_RK.dat'
    H_RK_FPGA_Filename = 'FPGA_RK.dat'
    H_Cost_Reduction__FPGA_Filename = 'FPGA_1.dat'
    H_Cost_Reduction_MCU_Filename = 'MCU_1.dat'

class UpgradeProcessStatus: # 升级进度
    End = 0 #0-升级结束
    Ing = 1 #1-升级中
    Finish = 2#2.升级结束



class ModelId:
    """
    卡的ModelId
    """
    H_AUX_2U = 0x7B82
    Input_Mother_Child_Card = 0x7B00
    Input_4_3G_SDI_Card = 0x7B01
    Input_2_12G_SDI_Card = 0x7B02
    Input_HDMI20_DP12 = 0x7B03
    Input_2_DP11_Card = 0x7B04
    Input_IPC_Card = 0x7B05  # Input_2RJ45_1USB_Card = 0x7B05
    Input_4_HDBaseT_Card = 0x7B06
    H_2xHDMI20_I = 0x7B07
    H_2xHDMI20_2xDP12_I = 0x7B09  # 4选2输入卡
    H_2x_Fiber_I = 0x7B10  # 2路光纤
    H_1xST2110_I = 0x7B11  # 2110 卡
    H_2xHDMI21_2xDP14_AUDIO_I = 0x7B13  # 8k_30卡

    H_4xHDBaseT_Card = 0x7B0A # H4路 HDBaseT 输入卡
    H_1RJ45_I = 0x7B0B # H_1路 NDI 输入卡
    H_2xAudio_I_O = 0x7B0C #H2路音频输入2路音频输出卡
    H_2xAudio_I_Dante_Card = 0x7B0D #H_2路音频输入2路音频输出2路Dante卡
    H_4xHDMI_Audio_I = 0x7B0E  # 四路HDMI随路音频输入卡
    H_4xDVI_I = 0x7B16  # 四路DVI输入卡视频(不支持音频)
    H_2xHDMI_DP_I = 0x7B0F  # 四选二随路音频输入卡 31503 同一个modelId 2个子卡 2路DP1.2输入卡

    Output_4_DVI_Card = 0x7B40
    Output_4_HDMI_Card = 0x7B41
    Output_4_DP11_Card = 0x7B42
    Output_2_RJ45_Card = 0x7B44
    Output_Two_In_One_Card = 0x7B46 #16网口卡 31558
    Output_2RJ45_1HDMI_Card = 0x7B47
    Output_Two_In_One_Card_20 = 0x7B48 #20网口卡 31560
    Output_2_HDMI20_Card = 0x7B49 # 4k

    Output_4_Fiber_Card = 0x7B4A  # 4光纤卡 #4光口卡 31562
    Output_8_Fiber_Card = 0x7B4F  # 8光纤卡 8光口卡 31567
    Output_4_RJ45_Card =  0x7B4B # H_4路网口 视频输出卡
    Output_4_3G_SDI_Card = 0x7B4C  # H_4路 3G-SDI 输出卡
    Output_12_3G_SDI_Card = 0x7B4D  # H1路 12G-SDI 输出卡
    Output_4_HDBaseT_Card = 0x7B4E  # H_4路BaseT 视频输出卡

    Not_Online = 0x7501

    Pd_Input_Mother_Child_Card = 0x9000
    Pd_Input_4_3G_SDI_Card = 0x9001
    Pd_Input_2_12G_SDI_Card = 0x9002
    Pd_Input_HDMI20_DP12 = 0x9003
    Pd_Input_2_DP11_Card = 0x9004
    Pd_H_2xHDMI20_I = 0x9007
    Pd_H_2xHDMI20_2xDP12_I = 0x9009  # 4选2输入卡

    Pd_H_4xHDMI_Audio_I = 0x900E  # 四路HDMI随路音频输入卡
    Pd_H_2xHDMI_DP_I = 0x900F  # 四选二随路音频输入卡

    Pd_Output_4_DVI_Card = 0x9040
    Pd_Output_4_HDMI_Card = 0x9041
    Pd_Output_Two_In_One_Card = 0x9046 #16网口 平达系列
    Pd_Output_Two_In_One_Card_20 = 0x9048 #20网口 平达系列
    Pd_Output_2_HDMI20_Card = 0x9049  # 4k
    Pd_Output_4_RJ45_Card = 0x904B  # H_4路网口 视频输出卡

    #二合一子卡
    Two_In_One_Card_List = [Output_Two_In_One_Card, Output_Two_In_One_Card_20, Output_4_Fiber_Card, Output_8_Fiber_Card,Pd_Output_Two_In_One_Card, Pd_Output_Two_In_One_Card_20]
    DL_Output_Card_List = [Output_4_DVI_Card, Output_4_HDMI_Card, Output_4_HDBaseT_Card, Output_4_RJ45_Card, Pd_Output_4_DVI_Card, Pd_Output_4_HDMI_Card, Pd_Output_4_RJ45_Card]
    Input_SL_DL_List = [Input_Mother_Child_Card, H_4xHDBaseT_Card, H_4xHDMI_Audio_I,H_4xDVI_I, Pd_Input_Mother_Child_Card, Pd_H_4xHDMI_Audio_I]
    H_160T_Card_ModelId_List = [Input_Mother_Child_Card, Input_2_DP11_Card, H_2xHDMI20_I, Input_HDMI20_DP12, H_AUX_2U,
                                H_2xHDMI20_2xDP12_I, Pd_Input_HDMI20_DP12, Pd_H_2xHDMI20_I, Pd_H_2xHDMI20_2xDP12_I, Pd_Input_Mother_Child_Card, Pd_Input_2_DP11_Card]
    H_Cost_Reduction_Card_ModelId_List =[Output_4_HDMI_Card, Output_4_DVI_Card, Pd_Output_4_DVI_Card, Pd_Output_4_HDMI_Card]


class CardName:
    Unknown = 'unKnown'
    @staticmethod
    def get_card_name(model_id):
        card_name = CardName.Unknown
        if model_id == ModelId.H_AUX_2U:
            card_name = 'H_AUX_2U'
        elif model_id == ModelId.Input_Mother_Child_Card:
            card_name = 'Input_Mother_Child_Card'
        elif model_id == ModelId.Input_4_3G_SDI_Card:
            card_name = 'Input_4_3G_SDI_Card'
        elif model_id == ModelId.Input_2_12G_SDI_Card:
            card_name = 'Input_2_12G_SDI_Card'
        elif model_id == ModelId.Input_HDMI20_DP12:
            card_name = 'Input_HDMI20_DP12'
        elif model_id == ModelId.Input_2_DP11_Card :
            card_name = 'Input_2_DP11_Card'
        elif model_id == ModelId.Input_IPC_Card:
            card_name = 'Input_IPC_Card'
        elif model_id == ModelId.Input_4_HDBaseT_Card:
            card_name = 'Input_4_HDBaseT_Card'
        elif model_id == ModelId.H_2xHDMI20_I:
            card_name = 'H_2xHDMI20_I'
        elif model_id == ModelId.H_2xHDMI20_2xDP12_I:
            card_name = 'H_2xHDMI20_2xDP12_I'
        elif model_id == ModelId.Output_4_DVI_Card:
            card_name = 'Output_4_DVI_Card'
        elif model_id == ModelId.Output_4_HDMI_Card:
            card_name = 'Output_4_HDMI_Card'
        elif model_id == ModelId.Output_4_DP11_Card:
            card_name = 'Output_4_DP11_Card'
        elif model_id == ModelId.Output_4_HDBaseT_Card:
            card_name = 'Output_4_HDBaseT_Card'
        elif model_id == ModelId.Output_2_RJ45_Card:
            card_name = 'Output_2_RJ45_Card'
        elif model_id == ModelId.Output_4_Fiber_Card:
            card_name = 'Output_4_Fiber_Card'
        elif model_id == ModelId.Output_Two_In_One_Card:
            card_name = 'Output_Two_In_One_Card'
        elif model_id == ModelId.Output_2_HDMI20_Card:
            card_name = 'Output_2_HDMI20_Card'
        elif model_id == ModelId.Not_Online:
            card_name = 'Not_Online'
        elif model_id == ModelId.Output_4_RJ45_Card:
            card_name = 'Output_4_RJ45_Card'
        elif model_id == ModelId.Output_4_3G_SDI_Card:
            card_name = 'Output_4_3G_SDI_Card'
        elif model_id == ModelId.Output_12_3G_SDI_Card:
            card_name = 'Output_12_3G_SDI_Card'
        elif model_id == ModelId.H_4xHDBaseT_Card:
            card_name = 'H_4xHDBaseT_Card'
        elif model_id == ModelId.H_1RJ45_I:
            card_name = 'H_1RJ45_I'
        elif model_id == ModelId.H_2xAudio_I_O:
            card_name = 'H_2xAudio_I_O'
        elif model_id == ModelId.H_2xAudio_I_Dante_Card:
            card_name = 'H_2xAudio_I_Dante_Card'
        elif model_id == ModelId.H_4xHDMI_Audio_I:
            card_name = 'H_4xHDMI_Audio_I'
        elif model_id == ModelId.H_2xHDMI_DP_I:
            card_name = 'H_2xHDMI_DP_I'
        elif model_id == ModelId.H_4xDVI_I:
            card_name = 'H_4xDVI_I'
        elif model_id == ModelId.Output_8_Fiber_Card:
            card_name = 'Output_8_Fiber_Card'
        return card_name


class CardCategory:
    Unknown = 0
    TwoInOneCard = 1 # 二合一卡 16网口卡
    MotherChildCard = 2
    IPCCard = 3
    K4_HDMI_DP_Card = 4
    TwoInOneCard_20 = 5  # 二合一卡 20网口卡
    Output4DVICard = 6
    Output4HDMICard = 7
    Output4KCard = 8
    StdIOCard = 9
    Input2x4K = 10
    H_2xHDMI20_2xDP12_I = 11  # 4选2或者 单DP
    H_Output4LinkCard = 12 # 4网口卡
    Output4FiberCard = 15  # 4光纤卡
    Input4xHDBaseTCard = 16  #
    AudioCard = 17  #
    SDI3GCard = 18  #
    H4xHDMIAudioCard = 19  #
    Output4HDBaseTCard = 20
    SDI12GCard = 21
    H2xHDMIDPICard = 22
    H2xFIBERCard = 23
    H1xST2110Card = 24
    H1xNDICard = 25
    H_2xHDMI21DP14AUDIOCard = 26
    H_4xDVI = 27
    H_8xFiber_O_Card = 33 # 8光口输出卡(4光纤增强版)



    @staticmethod
    def get_card_category(model_id):
        card_category = CardCategory.Unknown
        if model_id == ModelId.Input_Mother_Child_Card:
            card_category = CardCategory.MotherChildCard
        elif model_id == ModelId.Pd_Input_Mother_Child_Card:
            card_category = CardCategory.MotherChildCard
        elif model_id == ModelId.Output_Two_In_One_Card:
            card_category = CardCategory.TwoInOneCard
        elif model_id == ModelId.Pd_Output_Two_In_One_Card:
            card_category = CardCategory.TwoInOneCard
        elif model_id == ModelId.Output_Two_In_One_Card_20:
            card_category = CardCategory.TwoInOneCard_20
        elif model_id == ModelId.Pd_Output_Two_In_One_Card_20:
            card_category = CardCategory.TwoInOneCard_20
        elif model_id == ModelId.Input_IPC_Card:
            card_category = CardCategory.IPCCard
        elif model_id == ModelId.Input_HDMI20_DP12:
            card_category = CardCategory.K4_HDMI_DP_Card
        elif model_id == ModelId.Pd_Input_HDMI20_DP12:
            card_category = CardCategory.K4_HDMI_DP_Card
        elif model_id == ModelId.Output_4_DVI_Card:
            card_category = CardCategory.Output4DVICard
        elif model_id == ModelId.Pd_Output_4_DVI_Card:
            card_category = CardCategory.Output4DVICard
        elif model_id == ModelId.Output_4_HDMI_Card:
            card_category = CardCategory.Output4HDMICard
        elif model_id == ModelId.Pd_Output_4_HDMI_Card:
            card_category = CardCategory.Output4HDMICard
        elif model_id == ModelId.Output_2_HDMI20_Card:
            card_category = CardCategory.Output4KCard
        elif model_id == ModelId.H_2xHDMI20_I:
            card_category = CardCategory.Input2x4K
        elif model_id == ModelId.Pd_H_2xHDMI20_I:
            card_category = CardCategory.Input2x4K
        elif model_id == ModelId.H_2xHDMI20_2xDP12_I:
            card_category = CardCategory.H_2xHDMI20_2xDP12_I
        elif model_id == ModelId.Pd_H_2xHDMI20_2xDP12_I:
            card_category = CardCategory.H_2xHDMI20_2xDP12_I
        elif model_id == ModelId.Output_4_Fiber_Card:
            card_category = CardCategory.Output4FiberCard
        elif model_id == ModelId.H_4xHDBaseT_Card:
            card_category = CardCategory.Input4xHDBaseTCard
        elif model_id == ModelId.H_2xAudio_I_O:
            card_category = CardCategory.AudioCard
        elif model_id == ModelId.Output_4_3G_SDI_Card:
            card_category = CardCategory.SDI3GCard
        elif model_id == ModelId.Output_12_3G_SDI_Card:
            card_category = CardCategory.SDI12GCard
        elif model_id == ModelId.Output_4_HDBaseT_Card:
            card_category = CardCategory.Output4HDBaseTCard
        elif model_id == ModelId.H_4xHDMI_Audio_I:
            card_category = CardCategory.H4xHDMIAudioCard
        elif model_id == ModelId.Pd_H_4xHDMI_Audio_I:
            card_category = CardCategory.H4xHDMIAudioCard
        elif model_id == ModelId.H_2xHDMI_DP_I:
            card_category = CardCategory.H2xHDMIDPICard
        elif model_id == ModelId.Pd_H_2xHDMI_DP_I:
            card_category = CardCategory.H2xHDMIDPICard
        elif model_id == ModelId.Output_4_RJ45_Card:
            card_category = CardCategory.H_Output4LinkCard
        elif model_id == ModelId.Pd_Output_4_RJ45_Card:
            card_category = CardCategory.H_Output4LinkCard
        elif model_id == ModelId.H_2x_Fiber_I:
            card_category = CardCategory.H2xFIBERCard
        elif model_id == ModelId.H_1xST2110_I:
            card_category = CardCategory.H1xST2110Card
        elif model_id == ModelId.Pd_Output_2_HDMI20_Card:
            card_category = CardCategory.Output4KCard
        elif model_id == ModelId.H_1RJ45_I:
            card_category = CardCategory.H1xNDICard
        elif model_id == ModelId.H_2xHDMI21_2xDP14_AUDIO_I:
            card_category = CardCategory.H_2xHDMI21DP14AUDIOCard
        elif model_id == ModelId.H_4xDVI_I:
            card_category = CardCategory.H_4xDVI
        elif model_id == ModelId.Output_8_Fiber_Card:
            card_category = CardCategory.H_8xFiber_O_Card


        return card_category


class FunctionType:
    SL = 0
    DL = 1
    HDMI20 = 2
    DP12 = 3
    HDMI21 = 4
    DP14 = 5
    Disable = 0xFF #255


class UpgradeItem(enum.Enum):
    """
    升级项目枚举
    """
    MCU = 1
    MCU_A = 2
    MCU_B = 3
    Aux_MCU = 4
    IMAGE = 5
    FNOT = 6
    FPGA = 7
    FPGA_A = 8
    FPGA_B = 9
    AuxFPGA_A = 10
    AuxFPGA_B = 11
    FPGA_A_RE = 12
    FPGA_B_RE = 13
    FPGA_RE = 21
    FPGA_COST = 23
    ASIC = 24



# H9 aux 替代料
class UpgradeH9AuxReplaceDataConfig:
    """
    硬件版本1.0 升级MCU.dat(1)、FPGA_A.dat(8)、FPGA_B.dat(9)
    硬件版本2.0 升级MCU.dat(1)、FPGA_A_1.dat(12)、FPGA_B_1.dat(13)

    2.RK IPC卡
    "chipBrief": RK3566 升级MCU_RK.dat(2)、FPGA_RK.dat(7)
    "chipBrief": HI3531 升级MCU.dat(1)、FPGA.dat(7)
    """
    DeviceModelIdList = [DeviceType.H_9U, DeviceType.Alita_9, DeviceType.P_H_9U]  # 0x750A
    HardWareVersion = ['2.0']
    HardWareVersionMapping = {
        '1.0': {
            'MCU': {
                'file': 'MCU.dat',
                'code': 1
            },
            'FPGA_A': {
                'file': 'FPGA_A.dat',
                'code': 8
            },
            'FPGA_B': {
                'file': 'FPGA_B.dat',
                'code': 9
            },
        },
        '2.0': {
            'MCU': {
                'file': 'MCU.dat',
                'code': 1
            },
            'FPGA_A': {
                'file': 'FPGA_A_1.dat',
                'code': 12
            },
            'FPGA_B': {
                'file': 'FPGA_B_1.dat',
                'code': 13
            },
        }

    }


# 4k输出
class Upgrade2HDMI20OutputDataConfig:
    HardWareVersion = ['2.0']
    HardWareVersionNum = 4
    dict_key = 'version'
    HardWareVersionMapping = {
        '1.0': {
            'MCU': {
                'file': 'MCU.dat',
                'code': 1
            },
            'FPGA': {
                'file': 'FPGA.dat',
                'code': 7
            }
        },
        '2.0': {
            'FPGA': {
                'file': 'FPGA_1.dat',
                'code': 21
            },
        },
        dict_key: {
            'MCU': {
                'file': 'MCU_1.dat',
                'code': 2
            },
            'FPGA': {
                'file': 'FPGA_1.dat',
                'code': 23
            }
        }

    }


class TestPatternEnum(enum.Enum):
    """
    测试画面枚举
    """
    Black = 0x0000  # 黑色
    Red = 0x0001  # 红色
    Green = 0x0002  # 绿色
    Blue = 0x0003  # 蓝色
    White = 0x0004  # 白色
    Close = 0xFFFF  # 关闭
    # 0x0005    竖条
    # 0x0006    横条
    # 0x0007    棋盘
    # 0x0100    水平红色渐变
    # 0x0101    水平绿色渐变
    # 0x0102    水平蓝色渐变
    # 0x0103    水平白色渐变
    # 0x0104    垂直红色渐变
    # 0x0105    垂直绿色渐变
    # 0x0106    垂直蓝色渐变
    # 0x0107    垂直白色渐变
    # 0x0200    白色横线
    # 0x0201    白色竖线
    # 0x0202    白色左斜
    # 0x0203    白色右斜
    # 0x0204    横竖线交叉
    # 0x0205    左斜右斜交叉
    # 0x0300    定位
    # 0xFFFE    自定义测试画面
    #


class ScreenDefaultValue:
    """
    屏幕参数默认值
    """
    ImageQuality = {
        "contrast": {
            "all": 50,
            "R": 50,
            "G": 50,
            "B": 50
        },
        "brightness": {
            "all": 50,
            "R": 50,
            "G": 50,
            "B": 50
        },
        "hue": 0,
        "saturation": 50
    }
    Gamma = 1
    Test_Pattern = {
        "testPattern": TestPatternEnum.Close.value,
        "bright": 2,
        "grid": 3,
        "speed": 2
    }


class UserType:
    SysUser = 0
    LicenseUser = 1


class IPCDefaultValue:
    """ipc源路数"""
    Slot_Max_Source_Count = 512
    Slot_Max_Channel_Count = 1024
    Source_Max_Channel_Count = 512
    Source_Channel_Create_Default_Count = 50
    Batch_Create_Default_Count = 50
    Group_Max_Source_Count = 1024
    Group_Max_Count = 50
    Export_Source_Count = 100
    Group_Max_Add_Source_Count = 1000
    # IPC拼接源默认通道数量
    MONTAGE_Channel_Num = 2

class RoleMaxCountValue:
    Role_Max_Count = 99
    Role_Root_Id = 1
    Role_Default_Id = 2


class TwoInOneCardData_1040:
    """
    二合一卡1040万规格
    """
    TimingMaxBandWidth_1 = 165000000
    TimingMaxBandWidth_2 = 165000000
    SlotMaxArea = 624600000
    SlotMaxWidth = 10240
    SlotMaxHeight = 10240
    SlotMinWidth = 800
    SlotMinHeight = 600

    # 二合一卡单个接口最大带载
    SingleOutputDefaultResolution = 1920 * 1080 * 60

    # 接口标准宽度
    SingleOutputStandardWidth = 2560
    # 二合一卡接口最大宽度
    SingleOutputMaxWidth = 2560


class FourFiberCardData:
    """
    四光纤卡独立模式规格
    """
    TimingMaxBandWidth_1 = 330000000
    TimingMaxBandWidth_2 = 330000000
    SlotMaxArea = 1248600000
    SlotMaxWidth = 16384
    SlotMaxHeight = 16384
    SlotMinWidth = 800
    SlotMinHeight = 600

    # 四光纤卡单个接口最大带载
    SingleOutputDefaultResolution = 1920 * 1080 * 60

    # 接口标准宽度
    SingleOutputStandardWidth = 2560
    # 四光纤卡接口最大宽度
    SingleOutputMaxWidth = 2560


class FourFiberCardCopyData:
    """
    四光纤卡复制模式规格
    """
    TimingMaxBandWidth_1 = 330000000
    TimingMaxBandWidth_2 = 330000000
    SlotMaxArea = 624300000
    SlotMaxWidth = 16384
    SlotMaxHeight = 16384
    SlotMinWidth = 800
    SlotMinHeight = 600

    # 四光纤卡单个接口最大带载
    SingleOutputDefaultResolution = 1920 * 1080 * 60

    # 接口标准宽度
    SingleOutputStandardWidth = 2560
    # 四光纤卡接口最大宽度
    SingleOutputMaxWidth = 2560

class FourFiberCardBackupData:
    """
    四光纤卡备份模式规格
    """
    TimingMaxBandWidth_1 = 330000000
    TimingMaxBandWidth_2 = 330000000
    SlotMaxArea = 624300000
    SlotMaxWidth = 16384
    SlotMaxHeight = 16384
    SlotMinWidth = 800
    SlotMinHeight = 600

    # 四光纤卡单个接口最大带载
    SingleOutputDefaultResolution = 1920 * 1080 * 60

    # 接口标准宽度
    SingleOutputStandardWidth = 2560
    # 四光纤卡接口最大宽度
    SingleOutputMaxWidth = 2560


class EightFiberCardData:
    """
    8光纤卡规格
    """
    TimingMaxBandWidth_1 = 420000000
    TimingMaxBandWidth_2 = 420000000
    SlotMaxArea = 1568000000
    SlotMaxWidth = 16384
    SlotMaxHeight = 16384
    SlotMinWidth = 800
    SlotMinHeight = 600

    # 单个接口最大带载
    SingleOutputDefaultResolution = 1920 * 1080 * 60

    # 接口标准宽度
    SingleOutputStandardWidth = 2560
    # 接口最大宽度
    SingleOutputMaxWidth = 2560


class TwoInOneCardData_1300:
    """
    二合一卡1300万规格
    """
    TimingMaxBandWidth_1 = 208000000
    TimingMaxBandWidth_2 = 165000000
    SlotMaxArea = 13120000 * 60
    SlotMaxWidth = 10752
    SlotMaxHeight = 10752
    SlotMinWidth = 800
    SlotMinHeight = 600

    # 二合一卡单个接口最大带载
    SingleOutputDefaultResolution = 1920 * 1080 * 60

    # 接口标准宽度
    SingleOutputStandardWidth = 2560
    # 二合一卡接口最大宽度
    SingleOutputMaxWidth = 2560

    TimingMaxBandWidth = 165000000


class Output4KCardData:
    TimingMaxBandWidth = 594000000
    SingleOutputTimingMaxBandWidth = 150000000
    # 接口标准宽度
    SingleOutputStandardWidth = 2560


class InputData:
    Group_Max_Count = 50


Output_DL_Timing_Default = {
    "horizontal": {"addrTime": 1920, "frontPorch": 88, "sync": 44, "syncPolarity": 0, "totalTime": 2200},
    "vertical": {"addrTime": 2160, "frontPorch": 8, "sync": 10, "syncPolarity": 0, "totalTime": 2250},
    "refreshRate": 30}


Output_SL_Timing_Default = {
    "horizontal": {"addrTime": 1920, "frontPorch": 88, "sync": 44, "syncPolarity": 0, "totalTime": 2200},
    "vertical": {"addrTime": 1080, "frontPorch": 4, "sync": 5, "syncPolarity": 0, "totalTime": 1125},
    "refreshRate": 60}


Max_Band_Capacity_Map = {

    "input": {
        ConnectCapacity.SL: 165000000,
        ConnectCapacity.DL: 297000000,
        ConnectCapacity.K4: 594000000
    },
    "output": {
        ConnectCapacity.SL: 165000000,
        ConnectCapacity.DL: 297000000,
        ConnectCapacity.K4: 594000000
    }
}


# Output_DL_Timing_Default = {
#     "horizontal": {"addrTime": 3840, "frontPorch": 176, "sync": 88, "syncPolarity": 0, "totalTime": 4400},
#     "vertical": {"addrTime": 1080, "frontPorch": 4, "sync": 5, "syncPolarity": 0, "totalTime": 1125},
#     "refreshRate": 60}

class MVRWindowType:
    Input = 0
    PGM = 1
    PVW = 2
    IPC = 3


class UpgradeCardType:
    RK = "RK3566"
    HS = "HI3531"


class UpgradeCardCode:
    HS_Code = 1
    Rk_Code = 2

class UpgradeFuncCode:
    func_code_fade_in_out = 4  # 淡入淡出
    func_code = 128
    func_code_enable = 128 # 旋转
    func_code_common = 0
    func_dynamic_subPixe_code = 64 # 动态子像素


class DataType:
    Menu_Type = 1
    Input_Type = 2
    Output_Type  = 3
    Screen_Type = 4
    Mvr_Type = 5
    User_Type = 6
    Role_Type = 7


# 权限功能菜单code值
class PermissionCode:
    Screen_Configuration = 1
    Screen_Edit = 2
    Pre_Supervision = 3
    Device_Setting = 5
    EDID_Edit = 6
    Device_Other = 11
    IPC_Administration = 12
    Delete_Screen = 17
    Screen_3D = 18
    Screen_HDR = 19
    Screen_Genlock = 20
    Device_Check = 21
    IPC_Edit = 27
    Role_Operation = 30
    Edit_Stream = 32
    Open_Genlock = 33
    Preset_Effects = 34
    Device_Backups = 35
    User_Operation = 43


class DeviceConfig:
    Device_Font_Count = 32 # 字体最大数量
    ExportSysSetting  = 1 # 导出系统配置
    ExportSDP  = 3 # 导出SDP数据

class OutputCardMode:
    IndependentMode = 2 # 独立模式
    CopyMode = 1 # 复制模式
    BackupMode = 0 # 备份模式
    OutputTwoInOneCard = 3 # 二合一卡 16网口卡
    OutputTwoInOneCard_20 = 4 # 二合一卡 20网口卡


class IpcVideoMode:
    SplicingMode = 0
    FourWayMode = 1
    SixteenWayMode = 2
    EightWayMode = 3
    TwoWayMode = 4


class NdiVideoMode:
    SplicingMode = 0
    OneWayMode = 1
    TwoWayMode = 2

class NdiDecodeNumber:
    OneWayNumber = 1
    TwoWayNumber = 2
    FourWayNumber = 4

class IpcDecodeNumber:
    OneWayNumber = 1
    TwoWayNumber = 2
    FourWayNumber = 4
    EightWayNumber = 8
    SixteenWayNumber = 16


class IPCSlotId:
    Invalid = 255


class IPCTemplate:
    template = ["\"填写说明：\n",
                "1. 请使用模板文件进行填写并确保文件未加密，不要删减本说明及标题栏，带 * 为必填项，不能为空；\n"
                "2. 非必填字段(不带 * )中：RTSP协议数据需要填写厂商(大华、海康、宇视、雄迈、中维世纪、天地伟业、巨峰、尚维、天视通、技威、YOOSEE或其他)、端口、主码流、辅码流；\n",
                "3. 信号源名称：长度64个字符以内，通道名称：长度15个字符以内；\n",
                "4. 规格字段只能填写：200W、400W、800W、1200W；类型字段只能填写：IPC、NVR；配置方式只能填写：ONVIF、RTSP；\"\n",
                "* 信号源名称,* 规格,* 类型,* 配置方式,厂商,用户名,密码,* IP地址,端口号,* 通道名称,主码流,辅码流\n"]

    template_en = ["\"Instructions：\n",
                "1. Please fill in this template and make sure the file is not encrypted. Do not delete the instructions and title bar. The items with * are required and cannot be empty.；\n"
                "2. For the optional fields (without *), you need to fill in the manufacturer  (Dahua, Hikvision, Uniview, Xiongmai, Jovision, Tiandy, Jufeng, Sunywo, Topsee, Gwell, YooSee and more), port number, main stream and sub stream for IPC of RTSP protocol.\n",
                "3. Signal name length: less than 64 characters, channel name length: less than 15 characters\n",
                "4. For the IPC specification field, only 200W, 400W,800W and 1200W can be filled.For the type field,only IPC and NVR can be filled.For the config mode, only ONVIF and RTSP can be filled.\"\n",
                "* Signal Name,* IPC Specification,* Type,* Config Method,Manufacturer,User Name,Password,* IP Address,Port Number,* Channel Name,Main Stream,Sub Stream\n"]

    template_tc = ["\"填寫說明：\n",
                   "1. 請使用範本檔進行填寫並確保檔未加密，不要刪減本說明及標題欄，帶 * 為必填項，不能為空；\n"
                   "2. 非必填字段(不帶 * )中：RTSP協議數據需要填寫廠商(大華、海康、宇視、雄邁、中維世紀、天地偉業、巨峰、尚維、天視通、技威、YOOSEE或其他)、端口、主碼流、輔碼流；\n",
                   "3. 信號源名稱：長度64個字元以內，通道名稱：長度15個字元以內；\n",
                   "4. 規格字段只能填寫：200W、400W、800W、1200W；類型字段只能填寫：IPC、NVR；配置方式只能填寫：ONVIF、RTSP；\"\n",
                   "* 信號源名稱,* 規格,* 類型,* 配置方式,廠商,用戶名,密碼,* IP地址,端口號,* 通道名稱,主碼流,輔碼流\n"]

    template_ko = ["\"작성 안내: \n",
                "1. 반드시 템플릿 파일을 사용하여 작성해 주시고, 파일에 암호가 설정되지 않았는지 확인해 주세요. 본 안내문과 제목 행은 삭제하지 마시고, * 표시된 항목은 필수 입력 사항으로 비워둘 수 없습니다；\n"
                "2. 선택 입력 항목( * 표시 없음) 중 RTSP 프로토콜 데이터는 제조사(DAHUA, 하이크비전, YUSHI, XIONGMAI, ZHONGWEISHIJI, TIANDIWEIYE, JUFENG, SHANGWEI, TIANSHITONG, JIWEI, YOOSEE 또는 기타), 포트, 메인 스트림, 서브 스트림을 기재해야 합니다；\n",
                "3. 신호원 이름: 최대 64자 이내, 채널 이름: 최대 15자 이내로 작성해 주세요；\n",
                "4. '규격' 항목은 200W, 400W, 800W, 1200W 중 하나만 기입 가능하며, '유형'은 IPC 또는 NVR만, '설정 방식'은 ONVIF 또는 RTSP만 기입 가능합니다；\"\n",
                "* 신호원 이름,* 규격,* 유형,* 설정 방식,제조사,사용자명,비밀번호,* IP주소,포트 번호,* 채널 이름,메인 스트림,서브 스트림\n"]

    template_dict = {
        LanguageMode.English : template_en,
        LanguageMode.Chinese : template,
        LanguageMode.Tc: template_tc,
        LanguageMode.Korean: template_ko,
    }

class IPCExportConst:
    Decode_Capacity = {
        1: "800W",
        2: "200W",
        3: "400W",
        4: "1200W"
    }
    IPC_Type = {
        0: "IPC",
        1: "NVR"
    }
    Source_Type = {
        0: "GB28181",
        1: "RTSP",
        2: "ONVIF"
    }
    Manufacturer = {
        -1: {
            0: "",
            1: "",
            3: "",
            4: "",
        },
        0: {
            0: "Dahua",
            1: "大华",
            3: "大華",
            4: "DAHUA",
        },
        1: {
            0: "Hikvision",
            1: "海康",
            3: "海康",
            4: "Hikvision",
        },
        2: {
            0: "Uniview",
            1: "宇视",
            3: "宇視",
            4: "Uniview",
        },
        3: {
            0: "Xiongmai",
            1: "雄迈",
            3: "雄邁",
            4: "Uniview",
        },
        4: {
            0: "Jovision",
            1: "中维世纪",
            3: "中維世紀",
            4: "Jovision",
        },
        5: {
            0: "Tiandy",
            1: "天地伟业",
            3: "天地偉業",
            4: "Tiandy",
        },
        6: {
            0: "Jufeng",
            1: "巨峰",
            3: "巨峰",
            4: "Jufeng",
        },
        7: {
            0: "Sunywo",
            1: "尚维",
            3: "尚維",
            4: "Sunywo",
        },
        8: {
            0: "Topsee",
            1: "天视通",
            3: "天視通",
            4: "Topsee",
        },
        9: {
            0: "Gwell",
            1: "技威",
            3: "技威",
            4: "Gwell",
        },
        10: {
            0: "YOOSEE",
            1: "YOOSEE",
            3: "YOOSEE",
            4: "YOOSEE",
        },
        255: {
            0: "Other",
            1: "其它",
            3: "其他",
            4: "기타",
        }
    }


class IPCImportConst:
    ONVIF = {
        "decodeCapacity": 0,
        "endIp": "",
        "ipList": [],
        "sourceName": "",
        "createTime": "",
        "ipcType": 0,
        "ip": "",
        "channelList": [
            {
                "channelIndex": 1,
                "channelName": "",
                "streamList": [
                    {
                        "streamIndex": 0,
                        "protocol": {
                            "type": 2,
                            "onvif": {
                                "mode": 0,
                                "onvifType": 0,
                                "onvifIp": "0",
                                "onvifUsrName": "",
                                "onvifPassWord": ""
                            }
                        }
                    },
                    {
                        "streamIndex": 1,
                        "protocol": {
                            "type": 2,
                            "onvif": {
                                "mode": 0,
                                "onvifType": 0,
                                "onvifIp": "0",
                                "onvifUsrName": "",
                                "onvifPassWord": ""
                            }
                        }
                    }
                ]
            }
        ],
        "deviceId": 0
    }
    RTSP = {
        "decodeCapacity": 0,
        "endIp": "",
        "ipList": [],
        "sourceName": "",
        "createTime": "",
        "ipcType": 0,
        "ip": "",
        "channelList": [
            {
                "channelIndex": 1,
                "channelName": "",
                "streamList": [
                    {
                        "streamIndex": 0,
                        "protocol": {
                            "type": 1,
                            "rtsp": {
                                "manufacturer": 255,
                                "rtspUrl": "",
                                "rtspIp": "",
                                "rtspPort": 0,
                                "rtspUsrName": "",
                                "rtspPassWord": "",
                                "mode": 0
                            }
                        }
                    },
                    {
                        "streamIndex": 1,
                        "protocol": {
                            "type": 1,
                            "rtsp": {
                                "manufacturer": 255,
                                "rtspUrl": "",
                                "rtspIp": "",
                                "rtspPort": 0,
                                "rtspUsrName": "",
                                "rtspPassWord": "",
                                "mode": 0
                            }
                        }
                    }
                ]
            }
        ],
        "deviceId": 0
    }
    Decode_Capacity = {
        "800W": 1,
        "200W": 2,
        "400W": 3,
        "1200W": 4
    }
    IPC_Type = {
        "IPC": 0,
        "NVR": 1
    }
    Source_Type = {
        "RTSP": 1,
        "ONVIF": 2
    }
    Manufacturer = {
        "大华": 0,
        "DAHUA": 0,
        "海康": 1,
        "HIKVISION": 1,
        "宇视": 2,
        "UNIVIEW": 2,
        "雄迈": 3,
        "XIONGMAI": 3,
        "中维世纪": 4,
        "JOVISION": 4,
        "天地伟业": 5,
        "Tiandy": 5,
        "巨峰": 6,
        "Jufeng": 6,
        "尚维": 7,
        "Sunywo": 7,
        "天视通": 8,
        "Topsee": 8,
        "技威": 9,
        "Gwell": 9,
        "YOOSEE": 10,
        "其它": 255,
        "OTHER": 255
    }
    Header_Count = 6

    Encoding_List = [None, "UTF-8", "GBK", "GB2312"]  # 读取导入文件的解码方式


class IPCType:
    IPC = 0
    NVR = 1

class IPCCardType:
    HS = 0
    RK = 1


class IPCSourceType:
    GB28181 = 0
    RTSP = 1
    ONVIF = 2


class IPCImportStatus:
    Importing = 0  # 导入中
    All_Success = 1  # 全部成功
    Part_Success = 2  # 部分成功


class IPCCheckRowConst:
    Success = ""
    Source_Name_Err = "信号源名称不合法"
    Decode_Capacity_Err = "规格不合法"
    IPC_Type_Err = "类型不合法"
    Source_Type_Err = "配置方式不合法"
    Manufacturer_Err = "厂商不合法"
    IP_Err = "ip不合法"
    Port_Err = "端口不合法"
    Main_URL_ERR = "主码流为空"
    Sub_URL_ERR = "辅码流为空"
    Channel_Name_Err = "通道名称不合法"
    Create_Source_Err = "创建失败"

    Source_Name_Length = 64
    Channel_Name_Length = 15

class outputModeType:
    Standard_Type = 1
    Customer_Type = 2


class SecurityMode:
    Normal = 1
    Safe = 2
    Default = Normal


class SecurityConfigType:
    SecurityModeConfig = 'securityModeConfig'
    LoginConfig = 'loginConfig'
    PwdConfig = 'pwdConfig'


class SecuritySessionExpiredType:
    FixedTime = 1
    UnfixedTime = 2
    UnOverdue = 3


NormalModeSettings = {
    # 登录配置
    'loginConfig': {
        'multiLoginAllow': 1,  # 同一账号多端登录: 0 不允许，1 允许
        'sessionExpireType': SecuritySessionExpiredType.UnOverdue,  # 1 固定过期， 2 滑动过期
        'sessionExpireTime': 30,  # 分钟
        'loginFailedCheck': {  # 登录失败校验
            'enable': 0,  # 是否启用校验 1 是,0 否
            'duration': 60,  # 登录校验周期(分钟)
            'showVerifyCodeTimes': 3,  # 失败N次出现验证码
            'lockUserTimes': 8,  # 失败N次锁定用户
        }
    },
    # 密码配置
    'pwdConfig': {
        'pwdLength': {  # 密码长度限制[5-50]
            'min': 5,
            'max': 20
        },
        'pwdRule': {  # 密码规则
            'nums': 0,  # 必须包含数字 1是,0否
            'lowerLetter': 0,  # 必须包含小写字母 1是,0否
            'upperLetter': 0,  # 必须包含大写 1是,0否
            'symbolCharacter': 0  # 必须包含特殊符号 1是,0否
        }

    }
}


SafeModeSettings = {
    # 登录配置
    'loginConfig': {
        'multiLoginAllow': 0,  # 同一账号多端登录: 0 不允许，1 允许
        'sessionExpireType': SecuritySessionExpiredType.UnfixedTime,  # 1 固定过期， 2 滑动过期
        'sessionExpireTime': 30,  # 分钟
        'loginFailedCheck': {  # 登录失败校验
            'enable': 1,  # 是否启用校验 1 是,0 否
            'duration': 60,  # 登录校验周期(分钟)
            'showVerifyCodeTimes': 2,  # 失败N次出现验证码
            'lockUserTimes': 7,  # 失败N次锁定用户
        }
    },
    # 密码配置
    'pwdConfig': {
        'pwdLength': {  # 密码长度限制[5-50]
            'min': 8,
            'max': 20
        },
        'pwdRule': {  # 密码规则
            'nums': 1,  # 必须包含数字 1是,0否
            'lowerLetter': 1,  # 必须包含小写字母 1是,0否
            'upperLetter': 1,  # 必须包含大写 1是,0否
            'symbolCharacter': 1  # 必须包含特殊符号 1是,0否
        }

    }
}
class UserStatus:
    """
    用户状态：活动、锁定、改密码
    """
    Active = 1
    Locked = 0
    PasswordChanged = 2


class Encrypt:
    Method = 'pbkdf2:sha256:100000'
    Salt_Len = 16


SN_Prefix = 'splicer'
Auth_Code_Expire_Time = 15 * 60
Audio_Support_Slot_ID = 7
Preset_Max_Limit = 256

class SafeConfig:
    IsNotSafe = 0
    IsSafe = 1

class HttpConfigType:
    DeviceHttpConfig = 'http_enable'
    Https_WebSocket_Port = 8081
    Http_WebSocket_Port = 8080

class UserLoginVerify:
    """用户登录验证"""
    UserActiveTime = 30*60
    VerifyCodeCount = 3

class LogConfig:
    Success = "operate success"

class DateConfig:
    FormatDate = '%Y-%m-%d %H:%M:%S'

class UserRoleType:
    Admin_Role = 1

class ReadSign:
    Is_ALL = 1
    Is_Not_ALL = 0

class Output12GSDIConst:
    LimitWidth = 3840
    LimitHeight = 2160
    LimitRefresh = 23


class PlanarConst:
    Bkg_And_Osd_Max_Count = {
        0x7525: 5,  # H_9U
        0x7524: 3,  # H_Ctrl_5U
        0x7523: 2,  # H_Ctrl_2U
        0x752C: 10,  # H_Ctrl_15U
        0x752D: 10,  # H_Ctrl_15U_PRO
    }

class PackagePwdConst:
    # 升级包密钥
    Secret_Key = "ptkLqneij80B8mVjdHYO15ioyHSxADPl"

    GCM_IV_Length = 12
    GCM_Tag_Length = 16


class DeviceInfoType:
    PackagePwdType = 'main_packagePwd'
    IsDisplayed = 'is_displayed'
    IsPresetId = 'is_preset_id'
    DefaultValue = -1  # 开关数据默认值

# 动态子像素
class DynamicSubpixel:
    Width_Step = 4  # 默认宽步进为4  对步进值,向上取整
    Height_Step = 2  # 默认高步进为2 对步进值,向上取整

    # 动态子像素的可选值
    Invalid  = 0 # 默认值 表示未开启
    Delta1_L = 1
    Delta1_R = 2
    Delta1_U = 3
    Delta1_D = 4
    RGGB     = 5
    Delta3_R = 6
    Delta3_L = 7
    Delta3_U = 8
    DeLta3_D = 9
    Delta4_R = 10
    Delta4_L = 11
    Delta4_U = 12
    DeLta4_D = 13

    # 动态子像素对应的宽度步进值
    Delta_Step_Width_Dict = {
        Delta1_L : 2,
        Delta1_R : 2,
        Delta1_U : 2,
        Delta1_D : 2,
        RGGB     : 2,
        Delta3_R : 2,
        Delta3_L : 2,
        Delta3_U : 3,
        DeLta3_D : 3,
        Delta4_R : 2,
        Delta4_L : 2,
        Delta4_U : 4,
        DeLta4_D : 4,
    }

    # 动态子像素对应的高度步进值
    Delta_Step_Height_Dict = {
        Delta1_L: 2,
        Delta1_R: 2,
        Delta1_U: 2,
        Delta1_D: 2,
        RGGB: 2,
        Delta3_R: 3,
        Delta3_L: 3,
        Delta3_U: 2,
        DeLta3_D: 2,
        Delta4_R: 4,
        Delta4_L: 4,
        Delta4_U: 2,
        DeLta4_D: 2,
    }

# 性能优化,批量查询
class Batch:
    Screen_Count = 20   # 屏幕批量查询的数量
    Layer_Count  = 30   # 图层批量查询的数量
    InputCrop_Count = 40 # 输入源的截取源 批量查询的数量

class ST2110Card:
    # ST2110子卡切换状态
    ModeStatus_Success = 0  # 切换成功
    ModeStatus_Fail = 1  # 切换失败
    ModeStatus_Ing = 2  # 切换中

if __name__ == '__main__':
    print(LayerType.Default)
