from H_9U.mao.impl import preset
from H_9U.util.cache import cacher
from H_9U.models.cachekey import CacheKey, CacheDuration


class PresetApi:

    def read_list(self, device_id, screen_id):
        """
        读取场景列表并加入缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :return:
        """
        key = CacheKey.preset_list(device_id, screen_id)
        return cacher.PresetCache.try_get_value(key, CacheDuration.Default, preset.preset_list_read, device_id, screen_id)

    def read_detail(self,device_id, screen_id, preset_id):
        """
        读取场景详情并加入缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param preset_id: 场景id
        :return: 场景详情
        """
        key = CacheKey.preset_detail(device_id, screen_id, preset_id)
        return cacher.PresetCache.try_get_value(key, CacheDuration.Default, preset.preset_full_detail_read, device_id, screen_id, preset_id)


    def write_create(self, device_id, screen_id, data):
        """
        创建场景
        清除场景详情缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 场景详细
        :return: 场景id
        """
        rs = preset.preset_create_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.PresetCache.delete(CacheKey.preset_list(device_id, screen_id))
            if 'presetId' in data:
                cacher.PresetCache.delete(CacheKey.preset_detail(device_id, screen_id, data['presetId']))
        return rs

    def write_delete(self, device_id, screen_id, preset_id, data):
        """
        删除场景
        清除场景列表缓存
        清除场景详情缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param preset_id: 场景id
        :param data:
        :return: 结果对象
        """
        rs = preset.preset_delete_write(device_id, screen_id, preset_id, data)
        if rs['status'] == 0:
            cacher.PresetCache.clear()
        return rs

    def read_general(self, device_id, screen_id, preset_id):
        """
        读取场景基本信息
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param preset_id: 场景id
        :return:
        """
        return preset.preset_general_read(device_id, screen_id, preset_id)

    def write_general(self, device_id, screen_id, preset_id, data):
        """
        设备场景基本信息
        清除场景列表缓存
        清除场景详情缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param preset_id: 场景id
        :param data: 场景信息
        :return: 结果对象
        """
        rs = preset.preset_general_write(device_id, screen_id, preset_id, data)
        if rs['status'] == 0:
            cacher.PresetCache.clear()
        return rs

    def write_play(self,device_id, screen_id, preset_id,data):
        """
        载入场景
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param preset_id: 场景id
        :param data: 载入场景
        :return: 结果对象
        """
        rs = preset.preset_play_write(device_id, screen_id, preset_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.clear()
            cacher.ScreenCache.clear()
            cacher.PresetCache.clear()
        return rs


    def read_group_list(self,device_id, screen_id):
        """
        读取场景分组列表并加入缓存
        :param device_id: 场景id
        :param screen_id: 屏幕id
        :return: 场景分组信息
        """
        key = CacheKey.preset_group_list(device_id, screen_id)
        return cacher.PresetCache.try_get_value(key, CacheDuration.Default, preset.preset_group_list_read, device_id, screen_id)

    def write_group_create(self,device_id, screen_id, data):
        """
        创建场景分组
        清除场景分组列表缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 分组信息
        :return: 结果对象
        """
        rs = preset.preset_group_create_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.PresetCache.delete(CacheKey.preset_group_list(device_id, screen_id))
        return rs

    def write_group_update(self, device_id, screen_id, data):
        """
        修改场景分组
        清除场景分组列表缓存
        清除场景组缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 分组信息
        :return: 结果对象
        """
        rs = preset.preset_group_update_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.PresetCache.clear()
        return rs

    def write_group_delete(self,device_id, screen_id, data):
        """
        删除场景分组
        清除场景分组列表缓存
        清除场景组缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 数据信息
        :return: 结果对象
        """
        rs = preset.preset_group_delete_write(device_id, screen_id, data)
        if rs['status'] == 0:
            group_id = data['presetGroupId']
            cacher.PresetCache.delete(CacheKey.preset_group_list(device_id, screen_id))
            cacher.PresetCache.delete(CacheKey.preset_group_node_list(device_id, screen_id, group_id))
        return rs

    def read_group_node_list(self, device_id, screen_id, group_id):
        """
        读取场景组信息并加入缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param group_id: 组id
        :return: 组信息
        """
        key = CacheKey.preset_group_node_list(device_id, screen_id, group_id)
        return cacher.PresetCache.try_get_value(key, CacheDuration.Default,preset.preset_group_node_list_read,device_id, screen_id, group_id)

    def write_group_node_add(self,device_id, screen_id, group_id, data):
        """
        场景如添加场景
        移除场景组缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param group_id: 组id
        :param data: 场景信息
        :return: 结果对象
        """
        rs = preset.preset_group_node_add_write(device_id, screen_id, group_id,data)
        if rs['status'] == 0:
            cacher.PresetCache.delete(CacheKey.preset_group_node_list(device_id, screen_id, group_id))
        return rs

    def write_group_node_delete(self,device_id, screen_id, group_id, data):
        """
        删除场景组
        清除场景分组列表缓存
        清除场景组缓存
        :param device_id: 设备id
        :param screen_id:
        :param group_id:
        :param data:
        :return:
        """
        rs = preset.preset_group_node_delete_write(device_id, screen_id, group_id,data)
        if rs['status'] == 0:
            cacher.PresetCache.delete(CacheKey.preset_group_list(device_id, screen_id))
            cacher.PresetCache.delete(CacheKey.preset_group_node_list(device_id, screen_id, group_id))
        return rs

    def write_poll(self,device_id, screen_id,data):
        """
        场景轮训
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 场景轮训
        :return: 结果对象
        """
        rs = preset.preset_poll_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def read_poll(self,device_id, screen_id):
        """
        读取场景轮训状态
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :return: 场景轮训状态
        """
        return preset.preset_poll_read(device_id, screen_id)

    def write_copy(self, device_id, screen_id, data):
        """
        场景复制
        清除场景列表缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 复制数据
        :return: 结果对象
        """
        rs = preset.preset_cory_write(device_id, screen_id,data)
        if rs['status'] == 0:
            cacher.PresetCache.delete(CacheKey.preset_list(device_id, screen_id))
        return rs

    def write_nontime_save(self, device_id, screen_id, data):
        """
        非实时上屏保存场景
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 场景信息
        :return: 结果对象
        """
        rs = preset.preset_nontime_create(device_id, screen_id, data)
        if rs['status'] == 0:
            preset_id = rs['data']['id']
            cacher.PresetCache.delete(CacheKey.preset_detail(device_id, screen_id, preset_id))
            cacher.PresetCache.delete(CacheKey.preset_list(device_id, screen_id))
            cacher.PresetCache.delete(CacheKey.preset_group_list(device_id, screen_id))
            cacher.PresetCache.delete(CacheKey.preset_screen(device_id, screen_id))
        return rs

    def read_preset_config(self, device_id):
        """
        读取场景特效切换模式
        :param device_id: 设备Id
        :return:  特效模式
        """
        return preset.preset_config_read(device_id)

    def write_preset_config(self, device_id, data):
        """
        全局特效切换 模式
        :param device_id: 设备Id
        :param data: 切换模式
        :return: 结果对象
        """
        return preset.preset_config_write(device_id, data)


    def write_collection_create(self,device_id, data):
        rs = preset.preset_collection_create_write(device_id,data)
        if rs['status'] == 0:
            cacher.PresetCache.delete(CacheKey.preset_collection_list(device_id))
        return rs

    def write_collection_update(self, device_id, data):
        rs = preset.preset_collection_update_write(device_id, data)
        if rs['status'] == 0:
            cacher.PresetCache.clear()
        return rs

    def write_collection_delete(self,device_id,data):
        """
        1.删除指定的场景组合
        2.清除场景组合列表缓存
        """
        rs = preset.preset_collection_delete_write(device_id, data)
        if rs['status'] == 0:
            cacher.PresetCache.delete(CacheKey.preset_collection_list(device_id))
        return rs

    def write_collection_del_all(self, device_id, data):
        """
        1.清空场景组合
        2.清除场景组合列表缓存
        """
        rs = preset.preset_collection_del_all_write(device_id,data)
        if rs['status'] == 0:
            cacher.PresetCache.delete(CacheKey.preset_collection_list(device_id))
        return rs

    def write_collection_play(self, device_id, data):
        """
        1.加载场景组合
        """
        return preset.preset_collection_play_write(device_id, data)

    def read_collection(self,device_id, data):
        """
        读取场景组合详情
        """
        return preset.preset_collection_info_write(device_id, data)

    def read_collection_list(self,device_id):
        """
        读取场景组合列表
        """
        return preset.preset_collection_list_read(device_id)

    def read_multi_detail(self,device_id, data):
        """
        批量查询多个场景详情
        """
        return preset.multi_detail_write(device_id, data)

preset_api = PresetApi()

