"""
Created on 2019年6月28日
@author: 刘益宗
mvrWindow接口.
对应3.5代接口协议中的 mvrWindow接口
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""

from H_9U.protocol.fncode import FnCode
from H_9U.models.result import transfer_midware_resp
from H_9U import controller as ctrl
from H_9U.util.logwrapper import log_wrapper


@log_wrapper
def mvr_windows_list_read(device_id, mvr_id):
    """
    获取指定MVR上的窗口列表
    :param device_id: 设备ID
    :param mvr_id: MVR ID
    :return: 参考文档MVRWindows
    """
    return transfer_midware_resp(ctrl.read(FnCode.MVRWINDOW_LIST, device_id, mvr_id))


@log_wrapper
def mvr_window_full_detail_read(device_id, mvr_id, window_id):
    """
    读MVR的窗口的全部信息
    :param device_id: 设备id
    :param mvr_id: mvr id
    :param window_id: 窗口id
    :return:参考文档 MVRWindowFullDetail
    """
    return transfer_midware_resp(ctrl.read(FnCode.MVRWINDOW_FULL_DETAIL, device_id, mvr_id, window_id))


@log_wrapper
def mvr_window_create_write(device_id, mvr_id, data):
    """
    创建窗口
    :param device_id: 设备号
    :param mvr_id: MVR ID号
    :param data: 创建窗口具体信息MVRWindowCreate
    :return:参考文档create，返回Id
    """
    return transfer_midware_resp(ctrl.write(FnCode.MVRWINDOW_CREATE, data if isinstance(data, list) else [data], device_id, mvr_id))


@log_wrapper
def mvr_window_delete_write(device_id, mvr_id, window_id, data):
    """
    删除窗口
    :param device_id:设备号
    :param mvr_id:MVR ID号
    :param window_id:窗口号MVRWindowDelete
    :param data: jsondata
    :return:参考文档delete
    """
    return transfer_midware_resp(ctrl.write(FnCode.MVRWINDOW_DELETE, [data], device_id, mvr_id, window_id))


@log_wrapper
def mvr_window_general_write(device_id, mvr_id, window_id, data):
    """
    写窗口信息
    :param device_id: 设备号
    :param mvr_id: MVR ID号
    :param window_id: 窗口号
    :param data: json数据
    :return:参考文档 MVRWindowGeneral
    """
    return transfer_midware_resp(ctrl.write(FnCode.MVRWINDOW_GENERAL, [data], device_id, mvr_id, window_id))


@log_wrapper
def mvr_window_general_read(device_id, mvr_id, window_id):
    """
    写窗口信息
    :param device_id: 设备号
    :param mvr_id: MVR ID号
    :param window_id: 窗口号
    :return:参考文档MVRWindowGeneral
    """
    return transfer_midware_resp(ctrl.read(FnCode.MVRWINDOW_GENERAL, device_id, mvr_id, window_id))


@log_wrapper
def mvr_window_umd_enable_write(device_id, mvr_id, window_id, data):
    """
    设置窗口位置、UMD位置信息.
    :param data: UMD信息json
    :param device_id: 设备号
    :param mvr_id: MVR ID号
    :param window_id: 窗口号
    :return:参考文档MVRWindowUMDGeneral
    """
    return transfer_midware_resp(
        ctrl.write(FnCode.MVRWINDOW_UMD_ENABLE, [data], device_id, mvr_id, window_id))


@log_wrapper
def mvr_window_umd_enable_read(device_id, mvr_id, window_id):
    """
    读取UMD信息
    :param device_id: 设备号
    :param mvr_id: MVR ID号
    :param window_id: 窗口号
    :return:参考文档MVRWindowUMDGeneral
    """
    return transfer_midware_resp(ctrl.read(FnCode.MVRWINDOW_UMD_ENABLE, device_id, mvr_id, window_id))


@log_wrapper
def mvr_window_umd_color_read(device_id, mvr_id, window_id):
    """
    窗口读取边框宽度
    :param device_id: 设备Id
    :param mvr_id:  mvr id
    :param window_id:  窗口id
    :return:  参考中间件接口文档MVRWindowUMDColor
    """
    return transfer_midware_resp(
        ctrl.read(FnCode.MVRWINDOW_UMD_COLOR, device_id, mvr_id, window_id))


@log_wrapper
def mvr_window_umd_color_write(device_id, mvr_id, window_id, params):
    """
    窗口设置umd 颜色
    :param device_id: 设备Id
    :param mvr_id:  mvr id
    :param window_id:  窗口id
    :param params:  颜色数据MVRWindowUMDColor
    :return:  结果对象
    """
    return transfer_midware_resp(
        ctrl.write(FnCode.MVRWINDOW_UMD_COLOR, [params], device_id, mvr_id, window_id))


@log_wrapper
def mvr_window_border_width_write(device_id, mvr_id, window_id, params):
    """
    窗口读取边框宽度
    :param device_id: 设备Id
    :param mvr_id:  mvr id
    :param window_id:  窗口id
    :param params:  边框数据MVRWindowBorderWidth
    :return:  结果对象
    """
    return transfer_midware_resp(
        ctrl.write(FnCode.MVRWINDOW_BORDER_WIDTH, [params], device_id, mvr_id, window_id))


@log_wrapper
def mvr_window_border_width_read(device_id, mvr_id, window_id):
    """
    窗口读取边框宽度
    :param device_id: 设备Id
    :param mvr_id:  mvr id
    :param window_id:  窗口id
    :return:  参考中间件接口文档MVRWindowBorderWidth
    """
    return transfer_midware_resp(
        ctrl.read(FnCode.MVRWINDOW_BORDER_WIDTH, device_id, mvr_id, window_id))


@log_wrapper
def mvr_window_border_color_read(device_id, mvr_id, window_id):
    """
    窗口读取边框颜色
    :param device_id: 设备Id
    :param mvr_id:  mvr id
    :param window_id:  窗口id
    :return:  结果对象MVRWindowBorderColor
    """
    return transfer_midware_resp(
        ctrl.read(FnCode.MVRWINDOW_BORDER_CORLOR, device_id, mvr_id, window_id))


@log_wrapper
def mvr_window_border_color_write(device_id, mvr_id, window_id, params):
    """
    窗口设置边框颜色
    :param device_id: 设备Id
    :param mvr_id:  mvr id
    :param window_id:  窗口id
    :param params: 颜色信息MVRWindowBorderColor
    :return:  结果对象
    """
    return transfer_midware_resp(
        ctrl.write(FnCode.MVRWINDOW_BORDER_CORLOR, [params], device_id, mvr_id, window_id))

@log_wrapper
def mvr_window_write_source(device_id, mvr_id, window_id, data):
    """
    窗口切源
    :param device_id: 设备Id
    :param mvr_id:  mvr id
    :param window_id:  窗口id
    :param params: 切源数据对象
    :return:  结果对象
    """
    return transfer_midware_resp(
        ctrl.write(FnCode.MVRWINDOW_SOURCE, [data], device_id, mvr_id, window_id))

@log_wrapper
def mvr_window_read_group_list(device_id, mvr_id):
    return transfer_midware_resp(
        ctrl.read(FnCode.MVR_GROUP_LLIST, device_id, mvr_id))


@log_wrapper
def mvr_window_group_create(device_id, mvr_id, data):
    return transfer_midware_resp(
        ctrl.write(FnCode.MVR_GROUP_CREATE, [data], device_id, mvr_id))


@log_wrapper
def mvr_window_group_delete(device_id, mvr_id, group_id, data):
    return transfer_midware_resp(
        ctrl.write(FnCode.MVR_GROUP_DELETE, [data], device_id, group_id, mvr_id))


@log_wrapper
def mvr_window_group_update(device_id, mvr_id, group_id, data):
    return transfer_midware_resp(
        ctrl.write(FnCode.MVR_GROUP_CREATE, [data], device_id, group_id, mvr_id))


@log_wrapper
def mvr_window_group_node_list(device_id, mvr_id, group_id):
    return transfer_midware_resp(
        ctrl.read(FnCode.MVR_GROUP_NODE_LIST, device_id, mvr_id, group_id))

@log_wrapper
def mvr_window_group_poll_read(device_id, mvr_id):
    return transfer_midware_resp(
        ctrl.read(FnCode.MVR_GROUP_POLL, device_id, mvr_id))


@log_wrapper
def mvr_window_write_group_play(device_id, data):
    return transfer_midware_resp(
        ctrl.write(FnCode.MVR_GROUP_PLAY,  [data], device_id))

@log_wrapper
def mvr_window_write_group_poll(device_id, group_id, enable, data):
    return transfer_midware_resp(
        ctrl.write(FnCode.MVR_GROUP_POLL, [data], device_id, group_id, enable))