'''
Created on 2019年8月8日
@author: GongQingBao
下行控制器：通讯器、协议解析器、管道的组装者，为使用者提供读写接口.
处理了以下内容：
    1.负责创建'Communicator/通讯器'、'Pipe/管道'、'协议解析器';
    2.为外界/Service提供'读、写、初始化状态'三个接口;
    3.'读、写'接口会根据'功能码/fnCode'选择匹配的通讯器;
说明：
    1.通讯器、管道两者没有约束关系，完全解耦.Controller是通讯器、协议解析器、管道的组装者,三者各司其职、互不影响、可分别由不同的人编写;
    2.Controller是通讯器的使用者，通讯器是协议解析器、管道的使用者;
'''

import threading
import time
import traceback

from flask import g

from H_9U.conf.syssettings import SysSettings
from H_9U.ctrl.down.createCommunicator import CreateCommunicator
from H_9U.ctrl.down.protocolParser import ProtocolParser
from H_9U.ctrl.down.selectCommunicator import SelectCommunicator
from H_9U.protocol.resUtil import getRsList
from H_9U.protocol.responseSatus import ResponseStatus
from H_9U.util.log import logger, log_info


class Controller(object):


    def __init__(self, parser=None, communicatorParam=None):
        """构造函数
        :param parser-协议解析器[可以为None],为空时采用内部默认解析器
        :param communicatorParam-创建'Communicator/通讯器'所需参数[可以为None],为空时采用内部规则,该参数结构定义：[{name,timeout,parser,callbackFunction}]
        """
        self.createNum_init = 3      # 初始化时创建通讯器的数量
        self.createNum_increase = 1   # 递增创建通讯器的数量
        self.communicator_maxNum = 50 # 通讯器数量上限

        self.parser = ProtocolParser() if parser == None else parser # 外界若未指定'协议解析器'则使用默认的解析器
        self.communicatorParam = communicatorParam

        self.creator = CreateCommunicator(SysSettings.ServerIP, SysSettings.ServerPort, self.parser)
        self.selector = SelectCommunicator(self.creator, self.createNum_increase, self.communicator_maxNum)

        self.communicatorList = []           # 存储所有'Communicator/通讯器'实例
        self.initStatus = self.createNum_init # 记录初始化状态[0:初始化完成、>0:未完成]

        # 在timer中创建'Communicator/通讯器',目的：不阻塞主进程
        self.thread = threading.Thread(target=self.run)
        self.thread.start()


    def run(self):
        """在线程中创建'Communicator/通讯器'
        """
        logger.info("Communicator初始化线程开始...")

        try:
            self.communicatorList.extend(self.creator.create(self.createNum_init))
            
            # 在timer中检测'Communicator/通讯器'是否初始化完成
            self.timeCheck = threading.Thread(target=self.runCheck)
            self.timeCheck.start()

        except Exception:
            msg = traceback.format_exc()
            logger.error('Communicator初始化线程时-异常:%s' % (msg))

        logger.info("Communicator初始化线程结束!")


    def runCheck(self):
        """在线程中检测'Communicator/通讯器'是否初始化完成.
        """
        if self.communicatorList == None or len(self.communicatorList) == 0:
            self.initStatus = 0
        else:
            while True:
                for temp in self.communicatorList:
                    if temp.pipe.linked:
                        self.initStatus = self.initStatus - 1

                if self.initStatus <= 0:
                    self.initStatus = 0
                    logger.info("controller初始化-完毕!!!")
                    break
                else:
                    logger.info("controller初始化-中......")

                time.sleep(2)


    def getInitStatus(self):
        """获得controller是否初始化完毕
        :return: boolean[True-初始化完毕、False-初始化进行中]
        """
        if self.initStatus <= 0:
            return True
        else:
            return False


    def read(self, fnCode, *params):
        """从中间件获取信息
        :param fnCode:功能码[非空]
        :param params:可变参数
        :return: {msg:"",status:0,data:jsonArray}
            msg:消息文本;
            status:传输状态;
            data:json对象数组
        """
        bln = self.__checkParams(params) # 验证输参数是否合法
        if bln == False:
            rs = getRsList(status=10)
            logger.error("read时params参数不可为空!fnCode:%i" % fnCode)
            return rs
        log_info("请求嵌入式3.5代协议 read,fnCode:%s", hex(fnCode))
        communicator = self.selector.select(g.token, fnCode, self.communicatorList, 0)

        if communicator != None:
            return communicator.sendReceiveData(fnCode, 0, None, True, 0, *params)
        else:
            logger.error("read服务还未就绪、Server initing...")
            return {'msg':'无可用链接,Server initing...', 'status':ResponseStatus.MIDDLEWARE_COMMUNICATION_FALURE, 'data':[]}


    def write(self, fnCode, data, *params):
        """向中间件写入信息
        :param fnCode:功能码[非空]
        :param data:待发送的json对象或数组[可为空None]
        :param params:可变参数
        :return: {msg:"",status:0,data:jsonArray}
            msg:消息文本;
            status:传输状态;
            data:json对象数组
        """
        bln = self.__checkParams(params) # 验证输参数是否合法

        if data == None or bln != True:
            rs = getRsList(status=10)
            logger.error("参数不可为空!fnCode:%i" % fnCode)
            return rs
        log_info("请求嵌入式3.5代协议 write,fnCode:%s", hex(fnCode))
        communicator = self.selector.select(g.token, fnCode, self.communicatorList, 1)

        if communicator != None:
            return communicator.sendReceiveData(fnCode, 1, data, True, 0, *params)
        else:
            logger.error("write服务还未就绪、Server initing...")
            return {'msg':'无可用链接,Server initing...', 'status':ResponseStatus.MIDDLEWARE_COMMUNICATION_FALURE, 'data':[]}


    def __checkParams(self, params):
        """验证read/write的params参数,如果传了None则报错
        :param params:待验证参数
        :return: 是否通过[True-验证通过、False-验证失败]
        """
        for arg in params:
            if arg == None:
                return False

        return True


