"""
Created on 2019年6月28日
@author: 刘益宗
screen api
参考前端接口文档 screen 部分
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
import copy
import time

from flask import Blueprint, g
from flask.globals import request
from flask.json import jsonify

from H_9U.api.device import device_detail
from H_9U.api.filter.filter import _filter_screen_list_data, role_filter_data_func
from H_9U.api.layer import screen_layer_detail_list
from H_9U.api.output import output_original_detail, output_write_test_pattern, output_read_all_list
from H_9U.api.preset import preset_read_list
from H_9U.api.websender import push_data, push_data_org
from H_9U.models.result import get_result_model, ResInfo
from H_9U.models.syncdataname import SyncDataName
from H_9U.models.sysconst import ScreenDefaultValue, ConnectCapacity, TestPatternEnum, DataType, PermissionCode, \
    PlanarConst, DeviceType, Batch, ModelId, Output12GSDIConst
from H_9U.service.bkg import bkg_svc
from H_9U.service.common import get_connect_capacity
from H_9U.service.layer import layer_svc
from H_9U.service.output import output_svc
from H_9U.service.role import rolesvc
from H_9U.service.screen import screen_svc
from H_9U.util.common import valid_json, valid_params
from H_9U.util.log import logger
from H_9U.util.logwrapper import api_log
from H_9U.util.permission_valid import valid_function_permission, get_code_list

screen_bp = Blueprint('screen', __name__)


@screen_bp.route('/readAllList', methods=['GET', 'POST'])
@api_log
def screen_read_all_list():
    """
    读取Screen列表
    :return: readAllList
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    user = g.user
    role_id = user['roleId']
    # 循环调用
    #rs = screen_list(device_id)
    # 批量调用
    rs = screen_list_batch(device_id)
    if rs['status'] == 0:
        role_rs, role_type = role_filter_data_func(role_id)
        rs['data'] = _filter_screen_list_data(role_type, role_rs['data']['screenList'], rs['data'])
    return jsonify(rs)


@screen_bp.route('/readDetail', methods=['GET', 'POST'])
@api_log
def screen_read_detail():
    """
    读取屏幕详情
    :return: 参考readDetail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id_list = values
    screen_detail_list =[]
    rs = get_result_model()
    for x in screen_id_list:
        detail_rs = screen_detail(device_id, x)
        if detail_rs['status'] ==0:
            screen_detail_list.append(detail_rs['data'])
    rs['data'] = screen_detail_list
    return jsonify(rs)


@screen_bp.route('/createScreen', methods=['GET', 'POST'])
@api_log
def screen_create():
    """
    创建Screen
    :return:参考文档createScreen
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    # 格式化屏幕参数
    values = valid_params(params, 'deviceId', 'screenInterfaces')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, interfaces = values

    # 获取当前登录用户的角色
    user = g.user
    role_id = user['roleId']
    role_rs = rolesvc.get_role_by_id(role_id)
    if role_rs['status'] == 0 and role_rs['data']:
        if role_rs['data']['role_type'] != 1:
            return jsonify(get_result_model(ResInfo.Role_Jurisdiction_Code))

    params['createTime'] = str(time.time())
    # 创建屏幕
    rs = screen_svc.create(device_id, params)
    if rs['status'] == 0:
        screen_id = rs['data']['id']
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        push_data(screen_detail, SyncDataName.Screen_Create, device_id, screen_id)
        from H_9U.api.device import genLock_status_read
        push_data(genLock_status_read, SyncDataName.Device_Read_GenLock_Status, device_id)
    return jsonify(rs)


@screen_bp.route('/deleteScreen', methods=['GET', 'POST'])
@api_log
def screen_delete():
    """
    删除Screen
    :return: 参考文档deleteScreen
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    # 接口权限校验 删除Screen  code = 17  DataType = 4
    rs = valid_function_permission(PermissionCode.Delete_Screen, DataType.Screen_Type, [screen_id])
    if rs['status'] != 0 :
        return jsonify(rs)
    rs = screen_svc.delete(device_id, screen_id)
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        push_data_org(SyncDataName.Screen_Delete, {'deviceId': device_id, 'screenId': screen_id})
    return jsonify(rs)


@screen_bp.route('/deleteScreenBatch', methods=['GET', 'POST'])
@api_log
def screen_delete_screen_batch():
    """
    批量删除屏幕
    :return:参考文档deleteScreenBatch
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    device_id = values
    model = params.get('model', 0)

    screen_rs = screen_list(device_id)
    if screen_rs['status'] != 0:
        return get_result_model(ResInfo.Middle_Data_Err)

    # 按照传入Id删除
    if model == 1:
        del_list = params.get('screenIdList')
    else:
        del_list = [x['screenId'] for x in screen_rs['data']]

    poll_screen = [s['screenId'] for s in screen_rs['data'] if s['presetPoll']['enable'] == 1]

    # 接口权限校验 删除Screen  code = 17  DataType = 4
    rs = valid_function_permission(PermissionCode.Delete_Screen, DataType.Screen_Type, del_list)
    if rs['status'] != 0:
        return jsonify(rs)

    del_rs = []
    for s_id in del_list:
        if s_id in poll_screen:
            del_rs.append({
                'deviceId': device_id,
                'screenId': s_id,
                'status': ResInfo.Screen_Is_Preset_Polling.value
            })
        else:
            d_rs = screen_svc.delete(device_id, s_id)
            del_rs.append({
                'deviceId': device_id,
                'screenId': s_id,
                'status': d_rs['status']
            })
    rs = get_result_model()
    rs['data'] = del_rs
    push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
    push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
    push_data(screen_list, SyncDataName.Screen_Clear_data, device_id)
    return jsonify(rs)


@screen_bp.route('/writeGeneral', methods=['GET', 'POST'])
@api_log
def screen_write_general():
    """
    设置指定Screen的基本信息
    :return: 参考文档writeGeneral
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    # 接口权限校验 设置Screen信息  code = 1  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Configuration, DataType.Screen_Type, [screen_id])
    if rs['status'] != 0:
        return jsonify(rs)

    rs = screen_svc.general_write(device_id, screen_id, params)
    if rs['status'] == 0:
        push_data(screen_detail, SyncDataName.Screen_Write_General, device_id, screen_id)
    return jsonify(rs)


@screen_bp.route('/ftb', methods=['GET', 'POST'])
@api_log
def screen_ftb():
    """
    设置屏幕黑屏ftb
    type  0:开启 1:关闭
    :return:参考文档ftb
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenIdList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id_list = values

    if len(screen_id_list) == 0:
        return jsonify(get_result_model(ResInfo.Params_Error))

    # 接口权限校验 设置Screen信息  code = 2  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Screen_Type, screen_id_list)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = screen_svc.ftb(device_id, screen_id_list, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Ftb, params)
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
    return jsonify(rs)


@screen_bp.route('/writeFreeze', methods=['GET', 'POST'])
@api_log
def screen_write_freeze():
    """
    设置屏幕冻结freeze
    enable 1：开启 0：关闭
    :return:参考文档freeze
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenIdList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id_list = values

    if len(screen_id_list) == 0:
        return jsonify(get_result_model(ResInfo.Params_Error))

    # 接口权限校验 设置Screen信息  code = 2  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Screen_Type, screen_id_list)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = screen_svc.write_freeze(device_id, screen_id_list, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Write_Freeze, params)
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
    return jsonify(rs)


def _screen_read_bkg(device_id, screen_id):
    """
    屏幕读取bkg
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return:  结果对象，参考readBKG
    """
    rs = screen_svc.screen_bkg_read(device_id, screen_id)
    if rs['status'] == 0 and rs['data']:
        bkg_id = rs['data']['bkgId']
        img_url = ''
        if bkg_id >= 0:
            bkg_rs = bkg_svc.bkg_read_detail(device_id)
            if bkg_rs['status'] == 0:
                img_url = bkg_rs['data']['data']['image']['imgUrl']
        rs['data']['imgUrl'] = img_url
    return rs


@screen_bp.route('/writeBKG', methods=['GET', 'POST'])
@api_log
def screen_write_bkg():
    """
    设置屏幕BKG
    :return:参考文档 writeBkg
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'bkgId', 'enable')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, bkg_id, enable = values

    # 接口权限校验 设置Screen信息  code = 2  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Screen_Type, [screen_id])
    if rs['status'] != 0:
        return jsonify(rs)
        # 获取主控modelId
    device_rs = device_detail(device_id)
    if device_rs['status'] != 0:
        return device_rs
    model_id = device_rs['data']['modelId']
    if model_id in DeviceType.P_Main_All:
        if enable == 1:
            check_rs = check_screen_osd_or_bkg_limit(device_id, screen_id, 'bkg')
            if check_rs['status'] != 0:
                return jsonify(check_rs)
    rs = screen_svc.screen_bkg_write(device_id, screen_id, params)
    if rs['status'] == 0:
        image_url = ''
        if enable == 1:
            from H_9U.api.bkg import bkg_image_url
            bkg_rs = bkg_image_url(device_id, bkg_id)
            if bkg_rs['status'] == 0 and bkg_rs['data'] and bkg_rs['data']['imageUrl']:
                image_url = bkg_rs['data']['imageUrl']
        params['imageUrl'] = image_url
        push_data_org(SyncDataName.Screen_Write_BKG, params)
    return jsonify(rs)


@screen_bp.route('/writeOSD', methods=['GET', 'POST'])
@api_log
def screen_write_osd():
    """
    设置屏幕osd
    :return:参考文档writeOsd
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    from H_9U.util.common import valid_str_value
    for text in params['text']:
        if not valid_str_value(text['words'], 0, 1024):
            return jsonify(get_result_model(ResInfo.Screen_OSD_Words_Length_Err))

    # 接口权限校验 设置Screen信息  code = 2  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Screen_Type, [screen_id])
    if rs['status'] != 0:
        return jsonify(rs)
    device_rs = device_detail(device_id)
    if device_rs['status'] != 0:
        return device_rs
    model_id = device_rs['data']['modelId']
    if model_id in DeviceType.P_Main_All:
        if params['enable'] == 1:
            check_rs = check_screen_osd_or_bkg_limit(device_id, screen_id, 'osd')
            if check_rs['status'] != 0:
                return jsonify(check_rs)
    rs = screen_svc.osd_write(device_id, screen_id, params)
    if rs['status'] == 0:
        screen_detail = screen_svc.read_detail(device_id, screen_id)
        if screen_detail['status'] == 0 and screen_detail['data']:
            osd_data = {
                "deviceId": device_id,
                "screenId": screen_id,
                "Osd": screen_detail['data']['Osd'],
                "OsdImage": screen_detail['data']['OsdImage']
            }
            push_data_org(SyncDataName.Screen_Write_OSD, osd_data)
            rs['data'] = {'Osd': screen_detail['data']['Osd']}

    return jsonify(rs)


@screen_bp.route('/readOSD', methods=['GET', 'POST'])
@api_log
def screen_read_osd():
    """
    读取屏幕osd
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    rs = screen_svc.osd_read(device_id, screen_id)
    return jsonify(rs)


@screen_bp.route('/writeImageOSD', methods=['POST'])
@api_log
def screen_write_image_osd():
    """
    设置屏幕osd
    :return:参考文档writeOsd
    """
    params = request.form.get('data')
    if not params:
        params = valid_json(request.get_data().decode())

    params = valid_json(params)
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    # 接口权限校验 设置Screen信息  code = 2  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Screen_Type, [screen_id])
    if rs['status'] != 0:
        return jsonify(rs)

    has_file = False
    if 'file' in request.files:
        has_file = True
        f = request.files['file']
        if not f.filename:
            f.filename = params['image']['file']['fileName']

    rs = screen_svc.image_osd_write(device_id, screen_id, params, has_file)
    if rs['status'] == 0:
        detail = screen_svc.read_detail(device_id, screen_id)
        if detail['status'] == 0 and detail['data']:
            osd_data = {
                "deviceId": device_id,
                "screenId": screen_id,
                "Osd": detail['data']['Osd'],
                "OsdImage": detail['data']['OsdImage']
            }
            push_data_org(SyncDataName.Screen_Write_OSD, osd_data)
            rs['data'] = {"OsdImage": detail['data']['OsdImage']}
    return jsonify(rs)


@screen_bp.route('/writeOutputMode', methods=['GET', 'POST'])
@api_log
def screen_write_output_mode():
    """
    输出接口与屏幕拼接
    :return: 见文档writeOutputMode
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    # 接口权限校验 设置Screen信息  code = 1  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Configuration, DataType.Screen_Type, [screen_id])
    if rs['status'] != 0:
        return jsonify(rs)

    # 没有输出口信息，返回参数错误
    interfaces = params.get('screenInterfaces')
    if not interfaces:
        return jsonify(get_result_model(ResInfo.Params_Error))

    rs = screen_svc.write_output_mode(device_id, screen_id, params)
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        push_data(screen_detail, SyncDataName.Screen_Write_Output_Mode, device_id, screen_id)
    return jsonify(rs)


@screen_bp.route('/writeTestPattern', methods=['GET', 'POST'])
@api_log
def screen_write_test_pattern():
    """
    设置指定屏幕的测试画面信息
    :return:  参考文档writeTestPattern
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    # 接口权限校验 设置Screen信息  code = 1  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Configuration, DataType.Screen_Type, [screen_id])
    if rs['status'] != 0:
        return jsonify(rs)

    # 读取屏幕所有输出接口
    rs = screen_svc.get_screen_output_list(device_id, screen_id)
    if rs['status'] != 0 or not rs['data']:
        return jsonify(rs)
        # 查询有效输出口
    test_pattern_list = []
    is_success = True
    for output in [o for o in rs['data'] if 0 <= o['outputId'] < 255]:
        output_id = output['outputId']
        data = {
            "outputId": output_id,
            "deviceId": device_id,
            "testPattern": params['testPattern'],
            "bright": params['bright'],
            "grid": params['grid'],
            "speed": params['speed']
        }
        # 写测试画面
        rs = output_write_test_pattern(device_id, output_id, data)
        if rs['status'] == 0:
            test_pattern_list.append((device_id, output_id, data))
        else:
            is_success = False
            break
    # 测试画面没有全部成功
    if not is_success:
        # 关闭已开启的测试画面
        if test_pattern_list:
            for op in test_pattern_list:
                device_id, output_id, data = op
                data['testPattern'] = 0xFFFF
                output_write_test_pattern(device_id, output_id, data)
        rs = get_result_model(ResInfo.Screen_Test_Pattren_Error)

    push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
    push_data_org(SyncDataName.Screen_Write_TestPattern, params)
    return jsonify(rs)


@screen_bp.route('/writeImageQuality', methods=['GET', 'POST'])
@api_log
def screen_write_image_quality():
    """
    设置屏幕画质
    :return: 见文档writeImageQuality
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    # 接口权限校验 设置Screen信息  code = 1  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Configuration, DataType.Screen_Type, [screen_id])
    if rs['status'] != 0:
        return jsonify(rs)

    rs = screen_svc.screen_image_quality_write(device_id, screen_id, copy.deepcopy(params))
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        push_data_org(SyncDataName.Screen_Write_ImageQuality, params)
    return jsonify(rs)


@screen_bp.route('/writeGamma', methods=['GET', 'POST'])
@api_log
def screen_write_gamma():
    """
    设置屏幕gamma
    :return: 见文档writeGamma
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    # 接口权限校验 设置Screen信息  code = 1  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Configuration, DataType.Screen_Type, [screen_id])
    if rs['status'] != 0:
        return jsonify(rs)

    rs = screen_svc.screen_gamma_write(device_id, screen_id, copy.deepcopy(params))
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        push_data_org(SyncDataName.Screen_Write_Gamma, params)
    return jsonify(rs)


@screen_bp.route('writeHDR', methods=['GET', 'POST'])
@api_log
def screen_write_hdr():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.Screen_HDR, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    # 接口权限校验 设置Screen信息  code = 1  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Configuration, DataType.Screen_Type, [screen_id])
    if rs['status'] != 0:
        return jsonify(rs)

    rs = screen_svc.write_hdr(device_id, screen_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Write_HDR, params)
    return jsonify(rs)


@screen_bp.route('/write3D', methods=['GET', 'POST'])
@api_log
def screen_write_3d():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.Screen_3D, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    rs = valid_function_permission(PermissionCode.Screen_Configuration, DataType.Screen_Type, [screen_id])
    if rs['status'] != 0:
        return jsonify(rs)

    rs = screen_svc.write_3d(device_id, screen_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Write_3D, params)
    return jsonify(rs)


@screen_bp.route('/hasData', methods=['GET', 'POST'])
@api_log
def screen_has_data():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    from .layer import screen_layers
    from .preset import preset_origin_list
    layer_rs = screen_layers(device_id, screen_id)
    preset_rs = preset_origin_list(device_id, screen_id)

    has_data = False
    if layer_rs['status'] != 0 or preset_rs['status'] != 0:
        return jsonify(get_result_model(ResInfo.Middle_Data_Err))
    elif layer_rs['data']['screenLayers'] or preset_rs['data']['presets']:
        has_data = True

    rs = get_result_model()
    rs['data'] = {'hasData': has_data}
    return jsonify(rs)


@screen_bp.route('/nonTimeDetail', methods=['GET', 'POST'])
@api_log
def nontime_detail():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = screen_non_time_detail(device_id, screen_id)
    return jsonify(rs)


@screen_bp.route('/writeScreenGenlockPhaseOffset', methods=['GET', 'POST'])
@api_log
def write_genlock_phase_offset():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.Screen_Genlock, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    # 接口权限校验 设置Screen信息  code = 1  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Configuration, DataType.Screen_Type, [screen_id])
    if rs['status'] != 0:
        return jsonify(rs)

    rs = screen_svc.write_genlock_phase_offset(device_id, screen_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Write_Phase_Offset, params)
    return jsonify(rs)


@screen_bp.route('/templateList', methods=['GET', 'POST'])
@api_log
def read_template_list():
    """
    读取屏幕布局模板列表数据
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = screen_svc.read_template_list(device_id)
    return jsonify(rs)


@screen_bp.route('/addTemplate', methods=['GET', 'POST'])
@api_log
def template_add():
    """
    添加屏幕自定义布局模板列表数据
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    template_id = params.get('id', None)
    rs = screen_svc.template_add(device_id, params)
    if rs['status'] == 0:
        if template_id is None:
            template_id = rs['data']['id']
            params['id'] = template_id
            rs['data'] = params
            push_data_org(SyncDataName.Screen_Add_Template, params)
        else:
            rs['data'] = params
            push_data_org(SyncDataName.Screen_Update_Template, params)
    return jsonify(rs)


@screen_bp.route('/deleteTemplate', methods=['GET', 'POST'])
@api_log
def template_delete():
    """
    添加屏幕自定义布局模板列表数据
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'id')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, template_id = values
    user = g.user
    role_id = user['roleId']
    rs = get_role_screen_data(screen_list(device_id), role_id)
    if rs['status'] == 0 and rs['data']:
            for screen in rs['data']:
                screen_id = screen["screenId"]
                preset_rs = preset_read_list(device_id, screen_id)
                if preset_rs['status'] == 0 and preset_rs['data']:
                    template_id_list = [x['layoutTemplateId'] for x in preset_rs['data']['presets']]
                    if template_id in template_id_list:
                        return jsonify(get_result_model(ResInfo.Screen_Layout_Delete_Err))
    rs = screen_svc.template_delete(device_id, template_id)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Delete_Template, params)
    return jsonify(rs)


@screen_bp.route('/brightAdjList', methods=['GET', 'POST'])
@api_log
def bright_adj_list():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = screen_svc.read_bright_adj_list(device_id, screen_id)
    return jsonify(rs)


@screen_bp.route('/brightAdjSettings', methods=['GET', 'POST'])
@api_log
def bright_adj_settings():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = screen_svc.write_bright_adj_list(device_id, screen_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Bright_Adj_Settings, params)
    return jsonify(rs)


@screen_bp.route('/writeAudioAttribute', methods=['GET', 'POST'])
@api_log
def write_audio_attribute():
    """
    设置屏幕的音频属性
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'audio')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, audio = values
    rs = screen_svc.write_audio_attribute(device_id, screen_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Write_Attribute, params)
    return jsonify(rs)


@screen_bp.route('/writeVolume', methods=['GET', 'POST'])
@api_log
def write_screen_volume():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = screen_svc.volume_write(device_id, screen_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Write_Volume, params)
    return jsonify(rs)


@screen_bp.route('/multiNonTimeDetail', methods=['GET', 'POST'])
@api_log
def multi_nontime_detail():
    """
    批量读取 屏幕非实时状态下所有图层信息
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenIdList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id_list = values
    rs = screen_svc.multi_nontime_detail_read(device_id, params)
    return jsonify(rs)


@screen_bp.route('/multiWriteBrightness', methods=['GET', 'POST'])
@api_log
def screen_multi_write_brightness():
    """
    设置屏幕Brightness
    :return: 见文档writeBrightness
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenIdList', 'operation')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id_list, brightness = values
    # 接口权限校验 设置Screen信息  code = 2  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Screen_Type, screen_id_list)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = screen_svc.multi_write_brightness(device_id, screen_id_list, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Brightness_Multi, params)
    return jsonify(rs)


@screen_bp.route('/multiSaveBrightness', methods=['GET', 'POST'])
@api_log
def screen_multi_save_brightness():
    """
    固化屏幕Brightness
    :return: 见文档writeBrightness
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'screenList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_list = values
    screen_id_list = []
    for x in screen_list:
        screen_id_list.append(x['screenId'])
        # 接口权限校验 设置Screen信息  code = 2  DataType = 4
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Screen_Type, screen_id_list)
    if rs['status'] != 0:
        return jsonify(rs)
    device_id = 0
    rs = screen_svc.multi_save_brightness(device_id, screen_list, params)
    return jsonify(rs)



def screen_layers_list(device_id):
    rs = screen_svc.read_list(device_id)
    if rs["status"] != 0 or not rs['data']:
        return rs
    if rs['data']:
        screens = rs['data'].get("screens", None)
        layers_list = []
        if screens:
            for screen in screens:
                screen_id = screen["screenId"]
                layers = screen_layer_detail_list(device_id, screen_id)
                if layers['status'] == 0:
                    layers_list.append({
                        'screenId': screen_id,
                        'screenLayers': layers['data']
                    })
                else:
                    logger.error("获得screenId:%i,screenId:%i屏幕详情-失败!结果中将缺少该screen信息."
                                 % (device_id, screen_id))

        rs['data'] = layers_list
    else:
        rs['data'] = []
    return rs


def screen_non_time_detail(device_id, screen_id):
    rs = screen_svc.nontime_detail_read(device_id, screen_id)
    return rs


def screen_detail(device_id, screen_id):
    """
    读取屏幕详细信息
    :param device_id: 设备Id
    :param screen_id: 屏幕id
    :return: 屏幕详细信息
    """
    # 读取屏幕详情
    rs = screen_svc.read_detail(device_id, screen_id)
    if rs['status'] != 0 or not rs['data']:
        return rs
    # 处理屏幕画质等相关信息
    test_pattern_default = build_test_pattern(device_id, rs['data'])

    rs['data']['testPattern'] = test_pattern_default if test_pattern_default else ScreenDefaultValue.Test_Pattern
    # 屏幕添加mayuse信息
    rs['data']['mayUse'] = 1

    # 屏幕图层信息
    __build_layer_data(device_id, rs, screen_id)

    # 批量查询屏幕的图层数据
    #rs['data']['screenLayers'] = __build_layer_list(device_id, rs['data'], screen_id)

    # bkg 信息
    rs['data']['Bkg']['imgUrl'],rs['data']['Bkg']['thumbnailImgUrl'] = __build_bkg_data(device_id, rs['data'])
    return rs


def build_test_pattern(device_id, rs):
    outputs = rs['outputMode']['screenInterfaces']
    test_pattern_default = None
    # 为每个输出加入slotId和ConnectCapacity信息, aoix aoiy
    test_pattern_close = False
    for output in outputs:
        o_id = output['outputId']
        interface_type = -1
        output_name = ''
        connect_capacity = ConnectCapacity.Default
        if o_id < 0 or o_id >= 255:
            continue

        # 循环调用嵌入式0301协议,存在性能问题
        output_rs = output_original_detail(device_id, o_id)
        if output_rs['status'] == 0 and rs:
            connect_capacity = output_rs['data']['connectCapacity']
            interface_type = output_rs['data']['interfaceType']
            output_name = output_rs['data']['general']['name']
            test_pattern = output_rs['data']['testPattern']
            if test_pattern['testPattern'] == TestPatternEnum.Close.value:
                test_pattern_close = True
            test_pattern_default = {
                'testPattern': test_pattern['testPattern'],
                'bright': test_pattern['bright'],
                'grid': test_pattern['grid'],
                'speed': test_pattern['speed']
            }
            if 'audioInterface' in output_rs['data']:
                output['isSupportAudio'] = output_rs['data']['isSupportAudio']
                output['audioInterface'] = output_rs['data']['audioInterface']
        output['outputName'] = output_name
        output['interfaceType'] = interface_type
        output['connectCapacity'] = connect_capacity
    if test_pattern_default and test_pattern_close:
        test_pattern_default['testPattern'] = TestPatternEnum.Close.value
    return test_pattern_default


def build_test_pattern_v2(screen_detail,output_map):
    """
    补全屏幕的output属性值,和测试画面数据
    不再查询嵌入式0301协议
    """
    outputs = screen_detail['outputMode']['screenInterfaces']
    test_pattern_default = None
    # 为每个输出加入slotId和ConnectCapacity信息, aoix aoiy
    test_pattern_close = False
    for output in outputs:
        o_id = output['outputId']
        interface_type = -1
        output_name = ''
        connect_capacity = ConnectCapacity.Default
        if o_id < 0 or o_id >= 255:
            continue

        output_detail = output_map.get(o_id)
        if output_detail:
            connect_capacity = output_detail['connectCapacity']
            interface_type = output_detail['interfaceType']
            output_name = output_detail['general']['name']
            test_pattern = output_detail['testPattern']
            if test_pattern['testPattern'] == TestPatternEnum.Close.value:
                test_pattern_close = True
            test_pattern_default = {
                'testPattern': test_pattern['testPattern'],
                'bright': test_pattern['bright'],
                'grid': test_pattern['grid'],
                'speed': test_pattern['speed']
            }
        output['outputName'] = output_name
        output['interfaceType'] = interface_type
        output['connectCapacity'] = connect_capacity
    if test_pattern_default and test_pattern_close:
        test_pattern_default['testPattern'] = TestPatternEnum.Close.value
    return test_pattern_default



def __build_bkg_data(device_id, rs):
    bkg_id = rs['Bkg']['bkgId']
    bkg_url = ''
    thumbnail_img_url = ''
    if bkg_id == 255:
        return bkg_url,thumbnail_img_url
    if rs['Bkg']['enable'] == 0:
        return bkg_url,thumbnail_img_url
    bkg_rs = bkg_svc.bkg_read_detail(device_id, bkg_id)
    if bkg_rs['status'] == 0 and bkg_rs['data']:
        bkg_url           =  bkg_rs['data']['data']['image']['imgUrl']
        thumbnail_img_url = bkg_rs['data']['thumbnailData']['image']['imgUrl']
    return bkg_url,thumbnail_img_url




def __build_layer_data(device_id, rs, screen_id):
    layers = screen_layer_detail_list(device_id, screen_id)
    rs['data']['screenLayers'] = []
    if layers['status'] == 0 and layers['data']:
        rs['data']['screenLayers'] = layers['data']

# 批量查询单个屏幕下的图层列表
def __build_layer_list(device_id, screen_data, screen_id):
    # 提取屏幕下的图层id
    layer_id_list = []
    for item in screen_data['screenLayers']:
        layer_id_list.append({
            "screenId": screen_id,
            "layerId": item["layerId"]
        })

    layer_list = []
    batch_size = Batch.Layer_Count
    for i in range(0, len(layer_id_list), batch_size):
        batch = layer_id_list[i:i + batch_size]
        if len(batch) > 0:
            layer_list_rs = layer_svc.read_batch_layer_detail(device_id, layer_id_list)
            if layer_list_rs['status'] == 0 and layer_list_rs['data']:
                layer_list.extend(layer_list_rs['data'][0]['screenLayers'])

    return layer_list

def screen_list(device_id):
    """
    读取屏幕列表
    :param device_id: 设备id
    :return: 屏幕列表
    """
    rs = screen_svc.read_list(device_id)
    if rs["status"] != 0 or not rs['data']:
        return rs
    if rs['data']:
        screens = rs['data'].get("screens", None)
        detail_list = []
        if screens:
            for screen in screens:
                screen_id = screen["screenId"]
                detail_rs = screen_detail(device_id, screen_id)
                if detail_rs['status'] == 0 and detail_rs['data']:
                    detail_list.append(detail_rs['data'])
                else:
                    logger.error("获得screenId:%i,screenId:%i屏幕详情-失败!结果中将缺少该screen信息." % (device_id, screen_id))
        rs['data'] = detail_list
    else:
        rs['data'] = []
    return rs


def screen_list_batch(device_id):
    # 获取屏幕id列表
    rs = screen_svc.read_list(device_id)
    if rs["status"] != 0 or not rs['data']:
        return rs
    screens = rs['data'].get("screens", None)
    if not screens:
        rs['data'] = []
        return rs
    rs['data'] = []
    screen_detail_list = []
    screens_id_list = []
    for screen in screens:
        screens_id_list.append( {
            "screenId": screen["screenId"]
        })
    if len(screens_id_list) == 0:
        return rs
    batch_size = Batch.Screen_Count
    # 批量获取屏幕详情(每20个查询一次)
    for i in range(0, len(screens_id_list), batch_size):
        batch_screen_id_list = screens_id_list[i:i + batch_size]
        batch_read_rs = screen_svc.batch_read_screen_detail(device_id,batch_screen_id_list)
        if batch_read_rs['status'] != 0:
            logger.error("获取屏幕详情-失败!req:%s" % batch_screen_id_list)
            continue
        screen_detail_list.extend(batch_read_rs['data'])

    # 获取所有output数据
    output_list_rs = output_svc.read_list(device_id)
    if output_list_rs["status"] != 0 or not output_list_rs['data']:
        return rs
    # list to map
    output_map = {}
    for output in output_list_rs['data']['outputs']:
        # 补全connectCapacity数据
        output['connectCapacity'] = get_connect_capacity(output['interfaceType'])
        if output['modelId'] in ModelId.Two_In_One_Card_List:
            output['connectCapacity'] = ConnectCapacity.K4
        output_map[output['outputId']] = output


    layer_id_list = []
    for item in screen_detail_list:
        # 不同子卡的屏幕输出接口处理
        handle_screen_detail(item)
        # 处理屏幕osd和亮度
        screen_svc.build_screen_detail(device_id,item,item['screenId'])
        # 处理屏幕画质等相关信息
        test_pattern_default = build_test_pattern_v2(item,output_map)

        item['testPattern'] = test_pattern_default if test_pattern_default else ScreenDefaultValue.Test_Pattern
        # 屏幕添加mayUse信息
        item['mayUse'] = 1

        # bkg 信息
        item['Bkg']['imgUrl'],item['Bkg']['thumbnailImgUrl'] = __build_bkg_data(device_id, item)

        # 提取所有的图层id
        for layer_item in item['screenLayers']:
            layer_id_list.append({
                "screenId": item['screenId'],
                "layerId": layer_item["layerId"]
            })

    # 批量查询所有屏幕的所有图层数据
    layer_list = []
    batch_size = Batch.Layer_Count
    for i in range(0, len(layer_id_list), batch_size):
        batch = layer_id_list[i:i + batch_size]
        if len(batch) > 0:
            layer_list_rs = layer_svc.read_batch_layer_detail(device_id, layer_id_list)
            if layer_list_rs['status'] == 0 and layer_list_rs['data']:
                layer_list.extend(layer_list_rs['data'])

    layer_list_map = {value["screenId"]: value for value in layer_list}
    for item in [x for x in screen_detail_list]:
        if layer_list_map.get(item['screenId']):
            item['screenLayers'] = layer_list_map.get(item['screenId'])['screenLayers']


    rs['data'] = screen_detail_list
    return rs


def handle_screen_detail(item):
    interfaces = item['outputMode']['screenInterfaces']
    # 有二合一卡接口
    tio_slots = [x['slotId'] for x in interfaces
                 if x['modelId'] in ModelId.Two_In_One_Card_List]
    if tio_slots:
        tio_slots = set(tio_slots)
        # new_inters 所有非二合一卡的接口
        new_inters = [x for x in interfaces if x['modelId']
                      not in ModelId.Two_In_One_Card_List]
        for slot_id in tio_slots:
            # 所有二合一卡的interface
            tios = [x for x in interfaces if
                    x['slotId'] == slot_id
                    and x['modelId'] in ModelId.Two_In_One_Card_List]
            tios.sort(key=lambda i: i['interfaceId'])

            if slot_id != 255:  # 非无效slot，正常二合一卡
                # 通slot下所有二合一卡接口,interfaceId从小到大排序
                if len(set([x['x'] for x in tios])) == 1 and len(set([x['cropx'] for x in tios])) == 1:
                    direction = 1
                else:
                    direction = 0
                new_inter = None
                for tio in tios:
                    if new_inter is None:  # 第一个接口
                        new_inter = tio
                    else:
                        # if new_inter['y'] == tio['y']:  # 水平
                        if direction == 0:
                            new_inter['cropx'] += tio['cropx']
                            new_inter['width'] += tio['width']
                        else:  # 垂直
                            new_inter['cropy'] += tio['cropy']
                            new_inter['height'] += tio['height']

                new_inters.append(new_inter)
            else:  # 无效接口
                # 二合一卡的起始下标
                index = 0
                for _ in range(len(tios) // 4):  # 包含二合一卡的个数
                    new_inter = None
                    for i in range(4):  # interfaceid，从小到打，四个一组
                        tio = tios[index]
                        if new_inter is None:
                            new_inter = tio
                        else:
                            if new_inter['y'] == tio['y']:  # 水平
                                new_inter['cropx'] += tio['cropx']
                                new_inter['width'] += tio['width']
                            else:  # 垂直
                                new_inter['cropy'] += tio['cropy']
                                new_inter['height'] += tio['height']
                        index += 1

                    new_inters.append(new_inter)
        item['outputMode']['screenInterfaces'] = new_inters
    # 有DL output
    interfaces = item['outputMode']['screenInterfaces']
    dl_interfaces = [x for x in interfaces
                     if x['modelId'] in ModelId.DL_Output_Card_List and x['isDL'] == 1]
    if dl_interfaces:
        # new_inters 所有非DL的接口
        new_inters = [x for x in interfaces if ('isDL' not in x or x['isDL'] != 1)]
        dl_interfaces.sort(key=lambda x: x['dlId'], reverse=True)
        for i in range(len(dl_interfaces) // 2):
            new_inter = dl_interfaces[i * 2]
            new_inter['cropx'] += dl_interfaces[i * 2 + 1]['cropx']
            new_inter['width'] += dl_interfaces[i * 2 + 1]['width']
            new_inters.append(new_inter)
        item['outputMode']['screenInterfaces'] = new_inters
    # 有12 G output
    SDI_12_slots = []
    for x in interfaces:
        if x['modelId'] == ModelId.Output_12_3G_SDI_Card and x['resolution'][
            'height'] >= Output12GSDIConst.LimitHeight \
                and x['resolution']['width'] >= Output12GSDIConst.LimitWidth \
                and x['resolution']['refresh'] >= Output12GSDIConst.LimitRefresh:
            SDI_12_slots.append(x['slotId'])
    if SDI_12_slots:
        SDI_12_slots = set(SDI_12_slots)
        # new_inters 所有非二合一卡的接口
        new_inters = [x for x in interfaces if x['modelId'] not in [ModelId.Output_12_3G_SDI_Card]]
        for slot_id in SDI_12_slots:
            # 所有二合一卡的interface
            k4s = [x for x in interfaces if
                   x['slotId'] == slot_id
                   and x['modelId'] in [ModelId.Output_12_3G_SDI_Card]]
            k4s.sort(key=lambda i: i['interfaceId'])

            if slot_id != 255:  # 非无效slot，正常二合一卡
                # 通slot下所有二合一卡接口,interfaceId从小到大排序

                new_inter = None
                for k4 in k4s:
                    if new_inter is None:  # 第一个接口
                        new_inter = k4
                    else:
                        # if new_inter['y'] == tio['y']:  # 水平
                        new_inter['cropx'] += k4['cropx']
                        new_inter['width'] += k4['width']

                new_inters.append(new_inter)
            else:  # 无效接口
                # 二合一卡的起始下标
                index = 0
                for _ in range(len(k4s) // 4):  # 包含二合一卡的个数
                    new_inter = None
                    for i in range(4):  # interfaceid，从小到打，四个一组
                        k4 = k4s[index]
                        if new_inter is None:
                            new_inter = k4
                        else:
                            new_inter['cropx'] += k4['cropx']
                            new_inter['width'] += k4['width']

                        index += 1

                    new_inters.append(new_inter)
        item['outputMode']['screenInterfaces'] = new_inters
    k4_slots = [x['slotId'] for x in interfaces
                if x['modelId'] in [ModelId.Output_2_HDMI20_Card, ModelId.Pd_Output_2_HDMI20_Card]]
    if k4_slots:
        k4_slot = set(k4_slots)
        # new_inters 所有非二合一卡的接口
        new_inters = [x for x in interfaces if
                      x['modelId'] not in [ModelId.Output_2_HDMI20_Card, ModelId.Pd_Output_2_HDMI20_Card]]
        for slot_id in k4_slot:
            # 所有二合一卡的interface
            k4s = [x for x in interfaces if
                   x['slotId'] == slot_id
                   and x['modelId'] in [ModelId.Output_2_HDMI20_Card, ModelId.Pd_Output_2_HDMI20_Card]]
            k4s.sort(key=lambda i: i['interfaceId'])

            if slot_id != 255:  # 非无效slot，正常二合一卡
                # 通slot下所有二合一卡接口,interfaceId从小到大排序

                new_inter = None
                for k4 in k4s:
                    if new_inter is None:  # 第一个接口
                        new_inter = k4
                    else:
                        # if new_inter['y'] == tio['y']:  # 水平
                        new_inter['cropx'] += k4['cropx']
                        new_inter['width'] += k4['width']

                new_inters.append(new_inter)
            else:  # 无效接口
                # 二合一卡的起始下标
                index = 0
                for _ in range(len(k4s) // 4):  # 包含二合一卡的个数
                    new_inter = None
                    for i in range(4):  # interfaceid，从小到打，四个一组
                        k4 = k4s[index]
                        if new_inter is None:
                            new_inter = k4
                        else:
                            new_inter['cropx'] += k4['cropx']
                            new_inter['width'] += k4['width']

                        index += 1

                    new_inters.append(new_inter)
        item['outputMode']['screenInterfaces'] = new_inters


def _build_screen_data(device_id, screen):
    """
    组装屏幕详细信息
    :param device_id: 设备Id
    :param screen: screen
    :return: 屏幕详细信息
    """
    if screen:

        # 处理屏幕画质等相关信息
        outputs = screen['outputMode']['screenInterfaces']
        test_pattern_default = None
        # 获取有效的输出口
        if outputs:
            # 为每个输出加入slotId和ConnectCapacity信息, aoix aoiy
            test_pattern_close = False
            for output in outputs:
                connect_capacity = ConnectCapacity.Default
                o_id = output['outputId']
                interface_type = -1
                output_name = ''
                if 0 <= o_id < 255:
                    output_rs = output_original_detail(device_id, o_id)
                    if output_rs['status'] == 0:
                        connect_capacity = output_rs['data']['connectCapacity']
                        interface_type = output_rs['data']['interfaceType']
                        output_name = output_rs['data']['general']['name']
                        test_pattern = output_rs['data']['testPattern']
                        if test_pattern['testPattern'] == TestPatternEnum.Close.value:
                            test_pattern_close = True
                        test_pattern_default = {
                            'testPattern': test_pattern['testPattern'],
                            'bright': test_pattern['bright'],
                            'grid': test_pattern['grid'],
                            'speed': test_pattern['speed']
                        }

                output['outputName'] = output_name
                output['interfaceType'] = interface_type
                output['connectCapacity'] = connect_capacity

            if test_pattern_default and test_pattern_close:
                test_pattern_default['testPattern'] = TestPatternEnum.Close.value

        screen['testPattern'] = test_pattern_default if test_pattern_default else ScreenDefaultValue.Test_Pattern

        # 屏幕添加mayuse信息
        screen['mayUse'] = 1  # _screen_may_use(device_id, rs['data'])
        # 屏幕图层信息
        layers = screen_layer_detail_list(device_id, screen['screen_id'])
        screen['screenLayers'] = []
        if layers['status'] == 0 and layers['data']:
            screen['screenLayers'] = layers['data']

        # bkg 信息
        bkg_id = screen['Bkg']['bkgId']
        bkg_url = ''
        if bkg_id >= 0:
            bkg_rs = bkg_svc.bkg_read_detail(device_id, bkg_id)
            if bkg_rs['status'] == 0 and bkg_rs['data']:
                bkg_url = bkg_rs['data']['data']['image']['imgUrl']
        screen['Bkg']['imgUrl'] = bkg_url
    return screen


def screen_general_list(device_id):
    """
    屏幕基本信息列表
    :param device_id: 设备Id
    :return:  屏幕基本信息列表
    """
    rs = screen_svc.read_list(device_id)
    if rs["status"] == 0 and rs['data']:
        rs['data'] = rs['data']['screens']
    return rs


def screen_freezed_output(device_id):
    """
    被冻结、ftb的屏幕列表
    :param device_id: 设备id
    :return: 屏幕列表
    """
    return screen_svc.freezed_output(device_id)

def screen_freezed_output_v2(device_id):
    """
    被冻结、ftb的屏幕列表
    :param device_id: 设备id
    :return: 屏幕列表
    """
    return screen_svc.freezed_output_v2(device_id)

def screen_used_bkg(device_id):
    """
    屏幕使用的bkg列表
    :param device_id: 设备id
    :return: bkg列表
    """
    rs = screen_list(device_id)
    if rs['status'] == 0 and rs['data']:
        bkg_list = [x['Bkg']['bkgId'] for x in rs['data'] if x['Bkg']['enable'] == 1]
        rs['data'] = bkg_list
    return rs


def screen_clear_layers(device_id, screen_id):
    """
    清空屏幕图层
    :param device_id: 设备Id
    :param screen_id: 屏幕id
    :return: 结果对象
    """
    rs = screen_svc.screen_clear_layers(device_id, screen_id)
    return rs


def screen_notime_play(device_id, screen_id_list, data):
    """
    屏幕非实时上屏take
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param data: 图层信息
    :return: 结果对象
    """
    req_data_list = []
    for x in screen_id_list:
        req_params = {
            "deviceId": device_id,
            "screenId": x,
            "enNonTime": data.get('enNonTime'),
            "manualPlay":  data.get('manualPlay'),
        }
        req_data_list.append(req_params)
    rs = screen_svc.screen_notime_play(device_id, 0, req_data_list)
    return rs


def screen_read_hdr(device_id, screen_id):
    rs = screen_svc.read_hdr(device_id, screen_id)
    return rs


def get_role_screen_data(screen_rs, role_id):
    """
    获取当前登录用户的输入列表数据
    :return: 无分组信息
    """
    if screen_rs['status'] != 0 or not screen_rs['data']:
        return screen_rs
    screen_ids = []
    role_rs = rolesvc.get_role_by_id(role_id)
    if role_rs['status'] != 0 or not role_rs['data']:
        return role_rs
    role_type = role_rs['data']['role_type']
    if role_type != 1:
        # 查询当前角色拥有的输入源权限
        rs = rolesvc.get_screen_by_role_id(role_id)
        if rs['status'] == 0 and rs['data']:
            screen_ids = [x['screenId'] for x in rs['data']]
        all_message_list = []
        for all_message in screen_rs['data']:
            screen_id = all_message['screenId']
            if screen_id in screen_ids:
                all_message_list.append(all_message)
        screen_rs['data'] = all_message_list
    return screen_rs


def check_screen_osd_or_bkg_limit(device_id, screen_id, check_type):
    """
    校验所有屏幕已开启的osd/bkg数量是否超限
    :param device_id: 设备id
    :param screen_id: 屏幕id
    :param check_type: 校验的类型，osd/bkg
    :return: 屏幕osd列表
    """
    # 获取屏幕列表
    rs = get_result_model()
    screen_rs = screen_svc.read_list(device_id)
    if rs["status"] != 0:
        return screen_rs
    # 获取主控modelId
    device_rs = device_detail(device_id)
    if device_rs['status'] != 0:
        return device_rs
    model_id = device_rs['data']['modelId']
    screens = screen_rs['data'].get("screens", None)
    # 遍历列表，查询osd详情，判断osd数量是否超限
    if not screens:
        return get_result_model(ResInfo.Screen_List_Empty)
    count = 0
    for screen in screens:
        check_screen_id = screen["screenId"]
        # 校验屏幕osd/bkg限制时，跳过当前屏幕
        if check_screen_id == screen_id:
            continue
        if check_type == 'osd':
            detail = screen_svc.osd_read(device_id, check_screen_id)
        else:
            detail = screen_svc.screen_bkg_read(device_id, check_screen_id)
        if detail['status'] != 0:
            return detail
        if detail['data']['enable'] == 0:
            continue
        count += 1
        if count >= PlanarConst.Bkg_And_Osd_Max_Count[model_id]:
            return get_result_model(ResInfo.Screen_Support_Err)
    rs['data']['osdCount'] = count
    return rs




@screen_bp.route('/writeDynamicSubpixel', methods=['GET', 'POST'])
@api_log
def screen_write_dynamic_subpixel():
    """
    设置屏幕动态子像素[支持多屏批量设置]
    http://yapi.novatools.vip/project/613/interface/api/159658
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    device_id = params.get('deviceId')
    rs = screen_svc.screen_dynamic_subpixel_write(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Write_DynamicSubpixel, params)
    return jsonify(rs)